library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;

package AddressMap is

	constant CurrentVersion : std_logic_vector(31 downto 0) := x"20160522";

	constant NofModules : natural := 5;
	
    constant NofFBH_Raw : natural := 16;
    constant NofFBH     : natural := 31;
    constant NofCH      : natural := 64;
    constant NofTOF     : natural := 24;
    
    constant WidthCoin      : integer := 4;
    constant WidthIn        : integer := 6;
    constant WidthMtxCoin   : integer := 8;
    constant WidthOut       : integer := 10;
	
	subtype LocalAddressType is std_logic_vector(11 downto 0);
	subtype LocalBusInType   is std_logic_vector(23 downto 0);
	subtype LocalBusOutType  is std_logic_vector(7 downto 0);

	-- Module ID Map
	-- <Module ID : 31-28> + <Local Address 27 - 16>
	-- mid_LED:		0000
	-- mid_TDC:		0001
	-- mid_SEL:		0010
	-- mid_MTX3D:	0011
	-- mid_MTX2D:	0100
	-- mid_BCT:		1110
	-- SiTCP:		1111 (Reserved)

	-- Module ID Map -----------------------------------------------------------
	constant mid_LED 		: std_logic_vector(3 downto 0) := "0000";
	constant mid_TDC 		: std_logic_vector(3 downto 0) := "0001";
	constant mid_SEL 		: std_logic_vector(3 downto 0) := "0010";
	constant mid_MTX3D 		: std_logic_vector(3 downto 0) := "0011";
	constant mid_MTX2D 		: std_logic_vector(3 downto 0) := "0100";
	constant mid_BCT	 	: std_logic_vector(3 downto 0) := "1110";
	
	-- Local Address Map -------------------------------------------------------
	-- Module LED --
	constant LED_addr	        : LocalAddressType := x"000"; -- W/R, [3:0], LED On/Off & status 

	-- Module TDC --
	constant TDC_start	        : LocalAddressType := x"000"; -- W, execute send data
	constant TDC_lowth	        : LocalAddressType := x"001"; -- W, [7:0] search window
	constant TDC_highth	        : LocalAddressType := x"002"; -- W, [7:0] search window
	
	-- Module SEL --
	constant sel_nim1	        : LocalAddressType := x"000"; -- W, [3:0] NIM OUT 1
	constant sel_nim2	        : LocalAddressType := x"001"; -- W, [3:0] NIM OUT 2
	constant sel_nim3	        : LocalAddressType := x"002"; -- W, [3:0] NIM OUT 3
	constant sel_nim4	        : LocalAddressType := x"003"; -- W, [3:0] NIM OUT 4
	
	-- Module MTX3D/2D --
	constant MTX_low	        : LocalAddressType := x"000"; -- W, [15:0], lower 16bits
	constant MTX_high           : LocalAddressType := x"001"; -- W, [14:0], higher 15bits
	constant MTX_exec           : LocalAddressType := x"002"; -- W, Execute Shift Register
	
	-- BusController --
	constant BCT_Reset   		: LocalAddressType := x"000"; -- W
	constant BCT_Version 		: LocalAddressType := x"010"; -- R, [7:0] 4 byte (0x010,011,012,013);
	constant BCT_ReConfig  		: LocalAddressType := x"020"; -- W, Reconfig FPGA by SPI
	
	-- Local Address Map -------------------------------------------------------

end package AddressMap;

library ieee, work;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use work.AddressMap.all;

package AddressBook is

	subtype ModuleID is integer range -1 to NofModules-1;
	type Leaf is record
		ID : ModuleID;
	end record;

	type Binder is array (integer range <>) of Leaf;
	--constant i_LED		: Leaf := (ID => 0);
	constant i_LED		: Leaf := (ID => 0);
	constant i_TDC		: Leaf := (ID => 1);
	constant i_SEL		: Leaf := (ID => 2);
	constant i_MTX3D	: Leaf := (ID => 3);
	constant i_MTX2D	: Leaf := (ID => 4);
	constant i_Dummy	: Leaf := (ID => -1);

	constant AddressBook : Binder(NofModules-1 downto 0) :=
		( 0=>i_LED, 1=>i_TDC, 2=>i_SEL, 3=>i_MTX3D, 4=>i_MTX2D, others=>i_Dummy );

end package AddressBook;

library ieee, work;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use work.AddressMap.all;

package BusSignalTypes is

	type AddressArray is array (integer range NofModules-1 downto 0)
		of std_logic_vector(11 downto 0);
	type DataArray is array (integer range NofModules-1 downto 0)
		of std_logic_vector(7 downto 0);
	type ControlRegArray is array (integer range NofModules-1 downto 0)
		of std_logic;

	type BusControlProcessType is (
		Init,
		Idle,
		GetDest,
		SetBus,
		Connect,
		Finalize,
		Done
	);

	type BusProcessType is (
		Init,
		Idle,
		Connect,
		Write,
		Read,
		Execute,
		Finalize,
		Done
	);
	
	type SubProcessType is (
		SubIdle, 
		ExecModule,
		WaitAck,
		SubDone
	);
	
end package BusSignalTypes;
