module fifo_mod(
		ACTIVE_IN,
		OPERATE_CLK_IN,
		READOUT_CLK_IN,
		RESET_IN,
		INPUT_DATA,
		READ_MODE_IN,
		FIFO_RE_IN,
		OBDV_IN,
		FIFO_FORCE_WE,
		OUTPUT_DATA,
		FIFO_EMPTY_OUT,
		FIFO_FULL_OUT,
		DATA_VALID_OUT,
		FIFO_READ_READY
		);

   
   input 	    ACTIVE_IN;	    //
   input 	    RESET_IN;	    //
   input            OPERATE_CLK_IN; //
   input 	    READOUT_CLK_IN; //
   
   input [7:0] 	    INPUT_DATA;	    //
   input 	    READ_MODE_IN;   // 
   input 	    FIFO_RE_IN;	    //
   input 	    OBDV_IN;	    //
   input 	    FIFO_FORCE_WE;  // 
   
   output reg [7:0] OUTPUT_DATA;    //
   output reg 	    FIFO_EMPTY_OUT; //
   output reg 	    FIFO_FULL_OUT;  // 
   output reg 	    DATA_VALID_OUT; //
   output reg 	    FIFO_READ_READY;//

   reg 		    fifo_we_r;
   wire 	    fifo_we_s;
   
   wire [7:0] 	    fifo_in_s;
   wire [7:0] 	    fifo_out_s;

   wire 	    fifo_full_s;
   wire 	    fifo_empty_s;
   wire 	    data_valid_s;

   reg [7:0] 	    bus_shadowdata_r[0:2];
   
   assign fifo_we_s = (fifo_full_s == 1'b0) ?
		      (fifo_we_r | FIFO_FORCE_WE) : 1'b0;
   assign fifo_in_s[7:0] = (READ_MODE_IN == 1'b1) ?
			   bus_shadowdata_r[2] : INPUT_DATA[7:0];

   fifo fifo(
	     .rst(RESET_IN | ~ACTIVE_IN),  // IN   : Reset FIFO
	     .wr_clk(OPERATE_CLK_IN),      // IN   : Clock for write data 
	     .rd_clk(READOUT_CLK_IN),      // IN   : Clock for readout data
	     .din(fifo_in_s[7:0]),         // IN   : Input data
	     .wr_en(fifo_we_s),            // IN   : Input write enable
	     .rd_en(FIFO_RE_IN),           // IN   : Output read enable
	     .dout(fifo_out_s[7:0]),       // OUT  : Output data
	     .full(fifo_full_s),           // OUT  : FIFO full 
	     .empty(fifo_empty_s),         // OUT  : FIFO empty
	     .valid(data_valid_s)          // OUT  : Data Valid
	     );

   
   ///////////////////
   // Data shiifter
   //
   always @(negedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 bus_shadowdata_r[0] <= 8'd0;
	 bus_shadowdata_r[1] <= 8'd0;
	 bus_shadowdata_r[2] <= 8'd0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    bus_shadowdata_r[0] <= 8'd0;
	    bus_shadowdata_r[1] <= 8'd0;
	    bus_shadowdata_r[2] <= 8'd0;
   	 end else begin
	    
	    if(READ_MODE_IN == 1'b1)begin
	       bus_shadowdata_r[0] <= ~INPUT_DATA[7:0]; // Data input
	       bus_shadowdata_r[1] <= bus_shadowdata_r[0];
	       bus_shadowdata_r[2] <= bus_shadowdata_r[1];
	    end else begin
	       bus_shadowdata_r[0] <= 8'd0;
	       bus_shadowdata_r[1] <= 8'd0;
	       bus_shadowdata_r[2] <= 8'd0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
   
   /////////////////////////
   // Data valid by OBDV
   //
   reg [15:0]  obdv_r;
   
   always @(negedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 obdv_r[15:0] <= 16'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    obdv_r[15:0] <= 16'b0;
   	 end else begin

	    if(READ_MODE_IN == 1'b1)begin
	       obdv_r[15:0] <= {obdv_r[14:0], OBDV_IN};
	    end else begin
	       obdv_r[15:0] <= 15'b0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
   
   ////////////////////////////////////
   // Create write enable flag
   //
   always @(negedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 fifo_we_r <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    fifo_we_r <= 1'b0;
   	 end else begin

	    if(READ_MODE_IN == 1'b1)begin
	       if(obdv_r[3:0] == 4'b0010 || obdv_r[3:0] == 4'b0011)begin
		  fifo_we_r <= 1'b1;
	       end else if(obdv_r[3:0] == 4'b1000)begin
		  fifo_we_r <= 1'b0;
	       end
	    end else begin
	       fifo_we_r <= 1'b0;
	    end

   	 end
      end
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   wire ir_fifo_empty_s;
   double_ff dff_fifo_empty(
			    .idata(fifo_empty_s),
			    .oclk(OPERATE_CLK_IN),
			    .odata(ir_fifo_empty_s)
			    );

   
   //////////////////////////////////////////////////////////
   // To prevent to read before SVX finished to send data
   //
   reg fifo_read_ready_r;
   always @(negedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 fifo_read_ready_r <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    fifo_read_ready_r <= 1'b0;
   	 end else begin

	    if(ir_fifo_empty_s == 1'b1)begin
	       fifo_read_ready_r <= 1'b0;
	    end else if(READ_MODE_IN == 1'b1
			&& obdv_r[15:0] == 16'b1000_0000_0000_0000)begin
	       fifo_read_ready_r <= 1'b1;
	    end
	    
   	 end
      end
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   wire        ir_fifo_read_ready_s;
   double_ff dff_fifo_ready(
			  .idata(fifo_read_ready_r),
			  .oclk(READOUT_CLK_IN),
			  .odata(ir_fifo_read_ready_s)
			  );

   always @(posedge READOUT_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 FIFO_READ_READY <= 1'b0;
      end else begin
	 if(ACTIVE_IN == 1'b0)begin
	    FIFO_READ_READY <= 1'b0;
	 end else begin
	    FIFO_READ_READY <= ir_fifo_read_ready_s;
	 end
      end
   end
   
   ////////////////////////////////////
   // Data Output
   //
   always @(posedge READOUT_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 OUTPUT_DATA[7:0] <= 8'b0;
	 FIFO_EMPTY_OUT <= 1'b1;
	 FIFO_FULL_OUT <= 1'b0;
	 DATA_VALID_OUT <= 1'b0;
      end else begin
	 if(ACTIVE_IN == 1'b0)begin
	    OUTPUT_DATA[7:0] <= 8'b0;
	    FIFO_EMPTY_OUT <= 1'b1;
	    FIFO_FULL_OUT <= 1'b0;
	    DATA_VALID_OUT <= 1'b0;
	 end else begin
	    OUTPUT_DATA[7:0] <= fifo_out_s[7:0];
	    FIFO_EMPTY_OUT <= fifo_empty_s;
	    FIFO_FULL_OUT <= fifo_full_s;
	    DATA_VALID_OUT <= data_valid_s;
	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (negedge OPERATE_CLK_IN)
   
endmodule
