module conf_maker(
		  OPERATE_CLK_IN,
		  REG_CLK_IN,
		  ACTIVE_IN,
		  CONF_START_IN,
		  RESET_IN,
		  SIO_ADDR_IN,
		  SIO_WD_IN,
		  SIO_WE_IN,
		  PROUT_IN,
		  RAM_LOADED_OUT,
		  CONFIGURED_OUT,
		  CALSR_OUT,
		  PRIN_OUT,
		  FECLK_OUT,
		  ERROR_OUT
		  );
   
   input              OPERATE_CLK_IN; // Operational Clock
   input 	      REG_CLK_IN;     // Clock for Writing onto RAM
   input 	      ACTIVE_IN;      // Active Flag
   input 	      CONF_START_IN;  // Start Configuration (For CONFIGURE Command)
   input 	      RESET_IN;	      // System Reset
   input [11:0]       SIO_ADDR_IN;    // RAM's Address on which Data Written
   input [7:0] 	      SIO_WD_IN;      // Data written onto RAM
   input 	      SIO_WE_IN;      // RAM Write Enable
   input 	      PROUT_IN;	      // PRout from SVX4

   output reg 	      CALSR_OUT;      // Make signal to Latch into shadow register
   output reg 	      RAM_LOADED_OUT; // CtrlRAM is Ready
   output reg 	      CONFIGURED_OUT; // SVX is Configured completly
   
   output reg 	      PRIN_OUT;	      // PRin to SVX4
   output reg 	      FECLK_OUT;      // FECLK to SVX4
   
   output reg 	      ERROR_OUT;      // To inform failing to configure SVX4

   reg [79:0] 	      prout_holder_r;
   parameter [79:0]  CONF_HEAD_19M = {{4{10'b11111_00000}}, {4{10'b00000_11111}}};

   reg 		      configuring_r;
   wire 	      conf_reset_s;
   assign             conf_reset_s = ~ACTIVE_IN | RESET_IN;  // To Be Improved   
				   
   wire [7:0] 	      conf_rd_s;
   reg [11:0] 	      conf_r_addr_r;

   reg [11:0] 	      config_depth_r;
   parameter [7:0]    conf_trailer = 8'b1100_1100;

   wire 	      conf_re_s;
   assign             conf_re_s = (configuring_r == 1'b1
				   && RAM_LOADED_OUT == 1'b1
				   && prout_holder_r[79:0] != CONF_HEAD_19M
				   && conf_r_addr_r[11:0] < config_depth_r[11:0])
				   ? 1'b1 : 1'b0;
   
   conf conf(
	     // write data section
	     .clka(REG_CLK_IN),            // IN  : Input clock
	     .ena(ACTIVE_IN),              // IN  : BRAM enable
	     .wea(SIO_WE_IN),              // IN  : BRAM write enable
	     .addra(SIO_ADDR_IN[11:0]),    // IN  : register address written on
	     .dina(SIO_WD_IN[7:0]),        // IN  : Input register data

	     //readout data section
	     .clkb(OPERATE_CLK_IN),        // IN  : Output clock
	     .rstb(conf_reset_s),          // IN  : BRAM reset
	     .enb(conf_re_s),              // IN  : RAM read enable 
 	     .addrb(conf_r_addr_r[11:0]),  // IN  : RAM's address read
	     .doutb(conf_rd_s[7:0])        // OUT : Output register data
	     );

   
   /////////////////////////////////
   // Configure data depth
   //
   always @(posedge REG_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 config_depth_r[11:0] <= 12'd0;
	 RAM_LOADED_OUT <= 1'b0;
      end else begin
	 if(ACTIVE_IN == 1'b0)begin
	    config_depth_r[11:0] <= 12'd0;
	    RAM_LOADED_OUT <= 1'b0;
	 end else begin
	    
	    if(SIO_WE_IN == 1'b1
	       && SIO_WD_IN[7:0] == conf_trailer)begin
	       config_depth_r[11:0] <= SIO_ADDR_IN[11:0];
	       RAM_LOADED_OUT <= 1'b1;
	    end
	    
	 end
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge REG_CLK_IN or posedge RESET_IN)
   
   
   /////////////////////////////////
   // Reading Configure Bit Stream
   //
   reg [3:0] 	shift_counter_r;
   reg [7:0] 	prin_shifter_r;
   reg 		delay_prin_r;
   reg [7:0] 	feclk_shifter_r;

   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 conf_r_addr_r[11:0] <= 12'd1;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    conf_r_addr_r[11:0] <= 12'd1;
   	 end else begin
	    
	    if(conf_re_s == 1'b1)begin
	       // To Count Address of ConfigRAM Read
	       if(shift_counter_r[3:0] == 4'd0 || shift_counter_r[3:0] == 4'd2)begin
		  conf_r_addr_r[11:0] <= conf_r_addr_r[11:0] + 12'd1;
	       end else begin
		  conf_r_addr_r[11:0] <= conf_r_addr_r[11:0];
	       end
	    end else begin
	       conf_r_addr_r[11:0] <= 12'd1;
	    end // else: !if(prout_holder_r[79:0] != CONF_HEAD_19M...
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 shift_counter_r[3:0] <= 4'd0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    shift_counter_r[3:0] <= 4'd0;
   	 end else begin
	    
	    if(conf_re_s == 1'b1)begin
	       // To Serial Output of PRin and FECLK from ConfigRAM's data
	       if(shift_counter_r[3:0] < 4'd7)begin
		  shift_counter_r[3:0] <= shift_counter_r[3:0] + 4'd1; 
	       end else begin
		  shift_counter_r[3:0] <= 4'd0;
	       end
	    end else begin
	       shift_counter_r[3:0] <= 4'd0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 prin_shifter_r[7:0] <= 7'b0;
	 delay_prin_r <= 1'b0;
	 feclk_shifter_r[7:0] <= 8'd0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    prin_shifter_r[7:0] <= 7'b0;
	    delay_prin_r <= 1'b0;
	    feclk_shifter_r[7:0] <= 8'd0;
   	 end else begin
	    
	    if(conf_re_s == 1'b1)begin
	       delay_prin_r <= prin_shifter_r[7];
	       
	       if(shift_counter_r[3:0] == 4'd1)begin
		  // Reading PRin
		  prin_shifter_r[7:0] <= conf_rd_s[7:0];
		  feclk_shifter_r[7:0] <= {feclk_shifter_r[6:0], 1'b0};
	       end else if(shift_counter_r[3:0] == 4'd2)begin
		  // Reading FECLK
		  prin_shifter_r[7:0] <= {prin_shifter_r[6:0], 1'b0};
		  feclk_shifter_r[7:0] <= conf_rd_s[7:0];
	       end else begin
		  prin_shifter_r[7:0] <= {prin_shifter_r[6:0], 1'b0};
		  feclk_shifter_r[7:0] <= {feclk_shifter_r[6:0], 1'b0};
	       end
	    end else begin
	       prin_shifter_r[7:0] <= 7'b0;
	       delay_prin_r <= 1'b0;
	       feclk_shifter_r[7:0] <= 8'd0;
	    end // else: !if(prout_holder_r[79:0] != CONF_HEAD_19M...
	    
	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   
   /////////////////////////////////
   // OUTPUT
   //
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 PRIN_OUT <= 1'b0;
	 FECLK_OUT <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    PRIN_OUT <= 1'b0;
	    FECLK_OUT <= 1'b0;
   	 end else begin
	    
	    if(conf_re_s == 1'b1)begin
	       PRIN_OUT <= delay_prin_r;
	       FECLK_OUT <= feclk_shifter_r[7];
	    end else begin
	       PRIN_OUT <= 1'b0;
	       FECLK_OUT <= 1'b0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
   
   /////////////////////////////////////////
   // Receive Configure Bit Stream Header
   //
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 prout_holder_r[79:0] <= 80'd0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    prout_holder_r[79:0] <= 80'd0;
   	 end else begin
	    
	    if(configuring_r == 1'b1)begin
	       if(prout_holder_r[79:0] == CONF_HEAD_19M)begin
		  // Hold prout_holder_r
	       end else begin
		  prout_holder_r[79:0] <= {prout_holder_r[78:0], PROUT_IN};
	       end
	    end else begin
	       prout_holder_r[79:0] <= 80'd0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
   
   ///////////////////////////////////////////
   // Latch shadow register (in Init mode)
   //
   reg [7:0]    conf_sw_cnt_r;
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 conf_sw_cnt_r[7:0] <= 8'b0;
	 CALSR_OUT <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    conf_sw_cnt_r[7:0] <= 8'b0;
	    CALSR_OUT <= 1'b0;
   	 end else begin
	    
	    if(prout_holder_r[79:0] == CONF_HEAD_19M)begin
	       
   	       if(conf_sw_cnt_r[7:0] < 8'd10)begin
   		  conf_sw_cnt_r[7:0] <= conf_sw_cnt_r[7:0] + 8'd1;
   	       end else if(conf_sw_cnt_r[7:0] == 8'd10)begin
			   // && conf_sw_cnt_r[7:0] < 8'd11)begin
   		  conf_sw_cnt_r[7:0] <= conf_sw_cnt_r[7:0] + 8'd1;
   		  CALSR_OUT <= 1'b1;
   	       end else if(conf_sw_cnt_r[7:0] >= 8'd11
			   && conf_sw_cnt_r[7:0] < 8'd15)begin
   		  conf_sw_cnt_r[7:0] <= conf_sw_cnt_r[7:0] + 8'd1;
   		  CALSR_OUT <= 1'b0;
	       end
	       
	    end else begin
	       conf_sw_cnt_r[7:0] <= 8'b0;
	       CALSR_OUT <= 1'b0;
	    end // else: !if(prout_holder_r[79:0] == CONF_HEAD_19M)
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
   
   ///////////////////////////////////////////
   // CONFIGURED_OUT
   //
   wire 	ir_CONF_START_IN;
   double_ff d_ff_1(
		    .idata(CONF_START_IN),
		    .oclk(OPERATE_CLK_IN),
		    .odata(ir_CONF_START_IN)
		    );
   
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 CONFIGURED_OUT <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    CONFIGURED_OUT <= 1'b0;
   	 end else begin
	    
	    if(prout_holder_r[79:0] == CONF_HEAD_19M
	       && CONFIGURED_OUT == 1'b0
	       && conf_sw_cnt_r[7:0] == 8'd15)begin
	       CONFIGURED_OUT <= 1'b1;
	    end else if(ir_CONF_START_IN == 1'b1
			&& CONFIGURED_OUT == 1'b1)begin
	       CONFIGURED_OUT <= 1'b0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   
   /////////////////////////////////
   // Configuration start flag
   //
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 configuring_r <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    configuring_r <= 1'b0;
   	 end else begin
	    
	    if(ir_CONF_START_IN == 1'b1)begin
	       configuring_r <= 1'b1;
	    end else if(conf_sw_cnt_r[7:0] == 8'd15
			|| ERROR_OUT == 1'b1)begin
	       configuring_r <= 1'b0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   
   /////////////////////////////////
   // ERROR
   //
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 ERROR_OUT <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    ERROR_OUT <= 1'b0;
   	 end else begin
	    
	    if(prout_holder_r[79:0] != CONF_HEAD_19M
	       && conf_r_addr_r[11:0] == config_depth_r[11:0])begin
	       ERROR_OUT <= 1'b1;
	    end else if(ir_CONF_START_IN == 1'b1)begin
	       ERROR_OUT <= 1'b0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
endmodule
