module ctrl_sig_maker(
		      OPERATE_CLK_IN,
		      REG_CLK_IN,
		      ACTIVE_IN,
		      RESET_IN,
		      SIO_ADDR_IN,
		      SIO_WD_IN,
		      SIO_WE_IN,
		      TRIGGER_IN,
		      READ_START_IN,
		      READ_M_END_IN,
		      CONF_PARAM_IN,
		      RAM_LOADED_OUT,
		      TRIG_WAIT_OUT,
		      PARST_BUSY_OUT,
		      FECLK_CNT_OUT,
		      CTRL_SIG_OUT
		      );
   
   input              OPERATE_CLK_IN; // Operational Clock
   input 	      REG_CLK_IN;     // Clock for Writing onto RAM
   input 	      ACTIVE_IN;      // Active Flag
   input 	      RESET_IN;	      // Reset RAM (For UNCONFIGURE Command)
   input [13:0]       SIO_ADDR_IN;    // RAM's Address on which Data Written
   input [7:0] 	      SIO_WD_IN;      // Data written onto RAM
   input 	      SIO_WE_IN;      // RAM Write Enable
   input 	      TRIGGER_IN;     // Input Trigger
   input 	      READ_START_IN;  // Start Ctrl RAM Reading
   input 	      READ_M_END_IN;  // End Read Mode
   input [22:0]	      CONF_PARAM_IN;  // PickDel(configuration param)
   
   output reg 	      RAM_LOADED_OUT; // CtrlRAM is Ready
   output wire 	      TRIG_WAIT_OUT;  // Waiting for Trigger
   output wire 	      PARST_BUSY_OUT; // Preamp Reset Busy (For SOFIST Beamtest)
   output reg [3:0]   FECLK_CNT_OUT;  // FECLK's phase parameter
   output reg [15:0]  CTRL_SIG_OUT;   // Output Control Signal
   
   reg [13:0] 	      ctrl1_depth_r;
   reg [13:0] 	      ctrl2_depth_r;
   reg [13:0] 	      addr_counter_r;
   
   wire 	      ctrl_en_s;
   wire 	      ctrl_rst_s;
   
   wire [13:0] 	      ctrl1_addr_in_s;
   wire [13:0] 	      ctrl2_addr_in_s;
   wire [7:0] 	      ctrl1_wd_s;
   wire [7:0] 	      ctrl2_wd_s;
   wire 	      ctrl1_we_s;
   wire 	      ctrl2_we_s;

   wire 	      ctrl_re_s;
   wire [7:0] 	      ctrl1_sig_s;
   wire [7:0] 	      ctrl2_sig_s;

   
   //////////////////////////////////
   // Select writing data memory
   //
   wire [2:0] 	      ctrl_we_sw_s;

   function [1:0] write_ram_sw_f;
      input [2:0]     ram_sw_s;
      case(ram_sw_s)
	3'b101: write_ram_sw_f = 2'b01;
	3'b111: write_ram_sw_f = 2'b10;
	default: write_ram_sw_f = 2'b00;
      endcase // case (ram_sw_s)
   endfunction
   
   assign ctrl_we_sw_s = {ACTIVE_IN, SIO_ADDR_IN[0], SIO_WE_IN};
   assign {ctrl1_we_s, ctrl2_we_s} = write_ram_sw_f(ctrl_we_sw_s);

   assign ctrl1_addr_in_s[13:0] = (ctrl1_we_s == 1'b0) ? 14'd0
				  : {1'b0, SIO_ADDR_IN[13:1]};
   assign ctrl2_addr_in_s[13:0] = (ctrl2_we_s == 1'b0) ? 14'd0
				  : {1'b0, SIO_ADDR_IN[13:1]};
   
   assign ctrl1_wd_s[7:0] = (ctrl1_we_s == 1'b1) ? SIO_WD_IN[7:0] : 8'd0;
   assign ctrl2_wd_s[7:0] = (ctrl2_we_s == 1'b1) ? SIO_WD_IN[7:0] : 8'd0;

   assign ctrl_en_s = ACTIVE_IN && ~RESET_IN;
   assign ctrl_rst_s = ~ACTIVE_IN | RESET_IN;
   assign ctrl_re_s = READ_START_IN & RAM_LOADED_OUT;
   
   control1 control1(
		     // write data section
		     .clka(REG_CLK_IN),               // IN : CLK for Writing
		     .ena(ctrl_en_s),                 // IN : RAM Enable
		     .wea(ctrl1_we_s),                // IN : Write Enable
		     .addra(ctrl1_addr_in_s[13:0]),   // IN : Write Address
		     .dina(ctrl1_wd_s[7:0]),          // IN : Write Data
		     //readout data section
		     .clkb(OPERATE_CLK_IN),           // IN : CLK for Reading
		     .rstb(ctrl_rst_s),               // IN : RAM Reset
		     .enb(ctrl_re_s),                 // IN : Read Enable
		     .addrb(addr_counter_r[13:0]),    // IN : Read Address
		     .doutb(ctrl1_sig_s[7:0])         // OUT : Read Data
		     );

   control2 control2(
		     // write data section
		     .clka(REG_CLK_IN),               // IN : CLK for Writing
		     .ena(ctrl_en_s),                 // IN : RAM Enable
		     .wea(ctrl2_we_s),                // IN : Write Enable
		     .addra(ctrl2_addr_in_s[13:0]),   // IN : Write Address
		     .dina(ctrl2_wd_s[7:0]),          // IN : Write Data
		     //readout data section
		     .clkb(OPERATE_CLK_IN),           // IN : CLK for Reading
		     .rstb(ctrl_rst_s),               // IN : RAM Reset
		     .enb(ctrl_re_s),                 // IN : Read Enable
		     .addrb(addr_counter_r[13:0]),    // IN : Read Address
		     .doutb(ctrl2_sig_s[7:0])          // OUT : Read Data
		     );

   parameter [13:0] LOOP_BEGIN = 14'd22;
   parameter [13:0] LOOP_END = 14'd257;

   wire [13:0] 	      reset_wait_s;
   wire 	      pipe_scan_s;
   wire [5:0] 	      pickdel_s;
   wire [7:0] 	      latency_s;
   wire [7:0] 	      mask_len_s;

   assign pipe_scan_s = CONF_PARAM_IN[22];
   assign pickdel_s[5:0] = CONF_PARAM_IN[21:16];
   assign latency_s[7:0] = CONF_PARAM_IN[15:8];
   assign mask_len_s[7:0] = CONF_PARAM_IN[7:0];
   assign reset_wait_s[13:0] = {6'b0, pickdel_s[5:0], 2'b0} + {8'b0, pickdel_s[5:0]} // pickdel_s * 5
			       - {6'b0, latency_s[7:0]} + 14'd25;
   
   ///////////////////////////
   // Control Signal Depth
   //
   always @(posedge REG_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 // Initial State
	 RAM_LOADED_OUT <= 1'b0;
	 ctrl1_depth_r[13:0] <= 14'd0;
	 ctrl2_depth_r[13:0] <= 14'd0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    // Initial State
	    RAM_LOADED_OUT <= 1'b0;
	    ctrl1_depth_r[13:0] <= 14'd0;
	    ctrl2_depth_r[13:0] <= 14'd0;
   	 end else begin
	    
	    if(ctrl1_depth_r[13:0] == ctrl2_depth_r[13:0]
	       && ctrl1_depth_r[13:0] != 14'd0)begin
	       RAM_LOADED_OUT <= 1'b1;
	    end
	    
	    if(ctrl1_we_s == 1'b1
	       && ctrl1_wd_s[7:0] == 8'b1111_1111)begin
	       ctrl1_depth_r[13:0] <= ctrl1_addr_in_s[13:0];
	    end else if(ctrl2_we_s == 1'b1
			&& ctrl2_wd_s[7:0] == 8'b1111_1111)begin
	       ctrl2_depth_r[13:0] <= ctrl2_addr_in_s[13:0];
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   
   ///////////////////////////
   // Reading ControlRAM
   //
   reg      triggered_r;
   reg      trigger_wait_r;   
   reg 	    read_mode_end_r;

   assign TRIG_WAIT_OUT = trigger_wait_r & ~triggered_r;
   
   initial begin
      FECLK_CNT_OUT <= 4'd0;
   end
   
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 read_mode_end_r <= 1'b0;
	 triggered_r <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    read_mode_end_r <= 1'b0;
	    triggered_r <= 1'b0;
   	 end else begin
	    
	    if(TRIGGER_IN == 1'b1 && TRIG_WAIT_OUT == 1'b1)begin
	       triggered_r <= 1'b1;
	    end else if(trigger_wait_r == 1'b0)begin
	       triggered_r <= 1'b0;
	    end

	    if(READ_M_END_IN == 1'b1)begin
	       read_mode_end_r <= 1'b1;
	    end else if(TRIG_WAIT_OUT == 1'b1)begin
	       read_mode_end_r <= 1'b0;
	    end
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   reg parst_busy_r;
   assign PARST_BUSY_OUT = parst_busy_r;
   reg [3:0] parst_cnt_r;
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 // Initial State
	 addr_counter_r[13:0] <= 14'd0;
	 trigger_wait_r <= 1'b0;
	 parst_cnt_r[3:0] <= 4'd0;
	 parst_busy_r <= 1'b1;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    // Initial State
	    addr_counter_r[13:0] <= 14'd0;
	    trigger_wait_r <= 1'b0;
	    parst_cnt_r[3:0] <= 4'd0;
	    parst_busy_r <= 1'b1;
   	 end else begin
	    
	    if(ctrl_re_s == 1'b1)begin
	       if(addr_counter_r[13:0] == LOOP_BEGIN[13:0] + reset_wait_s[13:0])begin
		  addr_counter_r[13:0] <= addr_counter_r[13:0] + 14'd1;
		  if(parst_cnt_r[3:0] == 4'd0
		     || pipe_scan_s == 1'b1)begin
		     trigger_wait_r <= 1'b0;
		     parst_busy_r <= 1'b1;
		  end

	       end else if(addr_counter_r[13:0] == LOOP_BEGIN[13:0]
			   + reset_wait_s[13:0] + {6'b000000, mask_len_s[7:0]}/*reset_time*/)begin
		  addr_counter_r[13:0] <= addr_counter_r[13:0] + 14'd1;
		  trigger_wait_r <= 1'b1;
		  parst_busy_r <= 1'b0;
		  if(parst_cnt_r[3:0] < 4'd7)begin
		     parst_cnt_r[3:0] <= parst_cnt_r[3:0] + 4'd1;
		  end else begin
		     parst_cnt_r[3:0] <= 4'd0;
		  end

	       end else if(addr_counter_r[13:0] == LOOP_END[13:0] - 14'd1)begin
		  if(triggered_r == 1'b1)begin
		     addr_counter_r[13:0] <= addr_counter_r[13:0] + 14'd1;
		     trigger_wait_r <= 1'b0;
		     parst_busy_r <= 1'b1;
		  end else begin
		     // Loop waiting for Trigger
		     if(pipe_scan_s == 1'b1)begin
			addr_counter_r[13:0] <= LOOP_BEGIN[13:0] + 14'd5;
		     end else begin
			addr_counter_r[13:0] <= LOOP_BEGIN[13:0];
		     end
		  end

	       end else if(addr_counter_r[13:0] == 14'd676)begin
		  if(read_mode_end_r == 1'b1)begin
		     addr_counter_r[13:0] <= 14'd1767;
		     parst_cnt_r[3:0] <= 4'd0;
		  end else begin
		     addr_counter_r[13:0] <= 14'd667;
		  end
		  
	       end else if(addr_counter_r[13:0] == ctrl1_depth_r[13:0] - 14'd1)begin
		  addr_counter_r[13:0] <= 14'd1;
		  

	       end else if(trigger_wait_r == 1'b1
			   && FECLK_CNT_OUT[3:0] == (4'd4 - 4'd3))begin
		  if(triggered_r == 1'b1)begin
		     addr_counter_r[13:0] <= LOOP_END[13:0];
		     trigger_wait_r <= 1'b0;
		     parst_busy_r <= 1'b1;
		  end else begin
		     addr_counter_r[13:0] <= addr_counter_r[13:0] + 14'd1;
		  end
		  
	       end else begin
		  addr_counter_r[13:0] <= addr_counter_r[13:0] + 14'd1;
	       end
	       
	    end else begin
	       addr_counter_r[13:0] <= 14'd1;
	       trigger_wait_r <= 1'b0;
	       parst_busy_r <= 1'b1;
	       parst_cnt_r[3:0] <= 4'd0;
	    end // else: !if(ctrl_re_s == 1'b1)
	    
   	 end // else: !if(ACTIVE_IN == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)

   
   ///////////////////////
   // Signal Output
   //
   wire parst_s;
   assign parst_s = (parst_cnt_r[3:0] == 4'b0
		     || pipe_scan_s == 1'b1) ? ctrl2_sig_s[3] : 1'b0;
   
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 // Initial State
	 CTRL_SIG_OUT[15:11] <= 5'b00100;
	 CTRL_SIG_OUT[9:0] <= 10'b00_1100_1100;
	 if(FECLK_CNT_OUT[3:0] < 4'd4)begin
	    CTRL_SIG_OUT[10] <= 1'b0;
	    FECLK_CNT_OUT[3:0] <= FECLK_CNT_OUT[3:0] + 4'd1;
	 end else begin
	    CTRL_SIG_OUT[10] <= 1'b1;
	    FECLK_CNT_OUT[3:0] <= 4'd0;
	 end
      end else begin
	 if(ACTIVE_IN == 1'b0 || ctrl_re_s == 1'b0)begin
	    // Initial State
	    CTRL_SIG_OUT[15:11] <= 5'b00100;
	    CTRL_SIG_OUT[9:0] <= 10'b00_1100_1100;
	    if(FECLK_CNT_OUT[3:0] < 4'd4)begin
	       CTRL_SIG_OUT[10] <= 1'b0;
	       FECLK_CNT_OUT[3:0] <= FECLK_CNT_OUT[3:0] + 4'd1;
	    end else begin
	       CTRL_SIG_OUT[10] <= 1'b1;
	       FECLK_CNT_OUT[3:0] <= 4'd0;
	    end
	 end else begin
	    
	    CTRL_SIG_OUT[15:0] <= {ctrl1_sig_s[7:0],
				   ctrl2_sig_s[7:4], parst_s, ctrl2_sig_s[2:0]};
	    
	    if(CTRL_SIG_OUT[10]/*FECLK*/ == 1'b1)begin
	       FECLK_CNT_OUT[3:0] <= 4'd0;
	    end else begin
	       FECLK_CNT_OUT[3:0] <= FECLK_CNT_OUT[3:0] + 4'd1;
	    end
	    
	 end // else: !if(ACTIVE_IN == 1'b0 || ctrl_re_s == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
endmodule
