`define SYS_VERSION 32'h01_15_0315

module top(
	   //-----------------------------------
	   // SVX4
	   //-----------------------------------
	   //
	   // Bi-Directional LVDS
	   //
	   BUS,//   I/O  : 8bit BUSB
   
	   //
	   // Single ended Signal
	   //
	   PRin,//  OUT  : Priority InB
	   PRout,// IN   : Priority OutB

	   FECLK,// OUT  : FrontEnd ClockB
	   BECLK,// OUT  : BackEnd ClockB

	   OBDV,//  IN  : Odd Byte Data VaridB

	   PARST,//   OUT  : PreAmp Reset
	   PIPERD1,// OUT  : Pipeline Read 1
	   PIPERD2,// OUT  : Pipeline Read 2
	   L1A,//     OUT  : Level one Accept
	   CALSR,//   OUT  : Cal Strobe / Latch Shadow Register

	   FEMODE,//  OUT  : FrontEnd Mode
	   BEMODE,//  OUT  : BackEnd Mode
	   CHMODE,//  OUT  : Mode Change

	   //
	   // Bi Direction Enables
	   //
	   DDATA,//   OUT  : Data Drive Enable
	   RDATA,//   OUT  : Data Receive Enable

	   //---------------------------------------
	   // SEABAS FPGA(USER & SiTCP) CONNECTION
	   //---------------------------------------
	   //
	   // SEABAS IN/OUT Interface
	   //
	   CLK_50M,//       IN 
	   // SW_RSTn,//       IN
	   LED,//           OUT
	   DIP_SW,//        IN
	   NIM_OUT,
	   NIM_IN,

	   //
	   // User I/F
	   //
	   USR_CLK,//       OUT  : Clock
	   USR_ACTIVE,//    IN   : TCP established
	   USR_CLOSE_REQ,// IN   : Request to close a TCP Connection
	   USR_CLOSE_ACK,// OUT  : Acknowledge for USR_CLOSE_REQ

	   //
	   // TCP Tx : TCP Sender
	   //
	   USR_TX_AFULL,//  IN   : Almost full flag of a TCP Tx FIFO
	   USR_TX_WE,//     OUT  : TCP Tx Data write enable
	   USR_TX_WD,//     OUT  : TCP Tx Data[7:0]

	   //
	   // TCP Rx : TCP Receiver
	   //
	   USR_RX_EMPTY,//  IN   : Empty flag of a TCP RX flag
	   //USR_RX_RE,//   OUT  : TCP Rx Data Read enable
	   USR_RX_RV,//     IN   : TCP Rx data valid
	   USR_RX_RD,//     IN   : TCP Rx data[7:0]

	   ////////////////////////////////
	   // Only for simulation !!
	   // ////////////////////////////////
     	   SIO_ADDR,
    	   SIO_WD,
    	   SIO_WE,
	   // end for simulation //////////

	   ADC_PDWN, //     For saving power
	   USR_TESTPOINT,  // For Debug

	   //
	   // Register Access
	   //
	   REG_CLK,//       IN   : Clock
	   REG_ACT,//       IN   : Access Active
	   REG_DI,//        OUT  : Serial Data Output
	   REG_DO//         IN   : Serial Data Input
	   );

   /* ========================================================================== */
   
   ////////////////////////////////
   // Only for simulation !!
   ////////////////////////////////
   input [31:0] 	SIO_ADDR;
   input [7:0] 		SIO_WD;
   input 		SIO_WE;
   // end for simulation //////////
   
   /* ========================================================================== */  

   //-----------------------------------
   // SVX4
   //-----------------------------------
   inout [31:0] 	BUS;

   output [3:0] 	PRin;
   input [3:0] 		PRout;
   
   output [3:0] 	FECLK;
   output [3:0] 	BECLK;
   
   input [3:0] 		OBDV;

   output [3:0] 	PARST;
   output [3:0] 	PIPERD1;
   output [3:0] 	PIPERD2;
   output [3:0] 	L1A;
   output [3:0] 	CALSR;
   
   output [3:0] 	FEMODE;
   output [3:0] 	BEMODE;
   output [3:0] 	CHMODE;
   
   // ---------------------------
   
   reg [31:0] 		bus_r;
   
   reg [3:0] 		prin2_r;//   OUT
   wire [3:0] 		prout_s;//  IN
   reg [3:0] 		feclk2_r;// OUT
   wire 		beclk1_s;// OUT
   wire 		beclk2_s;// OUT
   wire [3:0] 		obdv_s;//   IN
   

   //-----------------------------------
   // SEABAS FPGA(USER & SiTCP) CONNECTION
   //-----------------------------------
   // Bi Direction Enable
   output [3:0] 	DDATA;
   output [3:0] 	RDATA;

   // SEABAS I/O Interface
   input 		CLK_50M;
   // input 		SW_RSTn;
   input [7:0] 		DIP_SW;
   output [7:0] 	LED;
   output [1:0] 	NIM_OUT;
   input [1:0] 		NIM_IN;
   
   wire [7:0] 		DIP_SW_s;
   assign               DIP_SW_s = DIP_SW;

   // User I/F
   output 		USR_CLK;
   input 		USR_ACTIVE;
   input 		USR_CLOSE_REQ;
   output 		USR_CLOSE_ACK;

   // TCP Tx
   input 		USR_TX_AFULL;
   output 		USR_TX_WE;
   output [7:0] 	USR_TX_WD;

   // TCP Rx
   input 		USR_RX_EMPTY;
   //output             USR_RX_RE;
   input 		USR_RX_RV;
   input [7:0] 		USR_RX_RD;

   // Register Access
   input 		REG_CLK;
   input 		REG_ACT;
   output 		REG_DI;
   input 		REG_DO;

   // For SEABAS's ADC power off
   output 		ADC_PDWN;
   assign               ADC_PDWN = DIP_SW_s[5];

   output [3:0] 	USR_TESTPOINT;

   // Assign using channel of D/B (1:use, 0:not use)
   wire [3:0] 		use_tele_s;
   assign use_tele_s[3:0] = DIP_SW_s[3:0];

   wire 		int_trig_mode_s;
   assign int_trig_mode_s = ~DIP_SW_s[7];

   
   //----------------------------------------
   // Create operation clocks for SVX4
   //----------------------------------------
   // DLL
   wire 		RST_DLL;
   wire  		dcm_rst_s;
   // assign               RST_DLL = ~SW_RSTn;
   assign               RST_DLL = DIP_SW_s[6];
   
   // CLKs created
   wire 		clk_50M_s;
   wire 		clk_50M_b; // From IBUFG output
   // wire 		clk_50M2_b; // From IBUFG output
   
   wire 		clk_40M_s; // main Operational clock
   wire 		clk_40M_90_s; // main Operational clock
   wire 		clk_40M_180_s; // main Operational clock
   wire 		clk_40M_270_s; // main Operational clock
   
   wire 		clk_25M_s; // For USR_TX
   wire 		clk_20M_s; // For USR_TX
   wire 		clk_19M_s; // For USR_TX

   wire  		locked_s;
   wire  		locked2_s;
   wire  		locked3_s;

   // =========== System Reset (asynchronous) ============
   wire 		sys_reset_s;
   assign sys_reset_s = ~locked_s;

   // Named Clock
   // wire 	conf_r_clk_s;
   // wire 	ctrl_r_clk_s;
   // wire 	fifo_w_clk_s;
   // wire 	fifo_r_clk_s;
   // assign conf_r_clk_s = clk_40M_s;
   // assign ctrl_r_clk_s = clk_40M_s;
   // assign fifo_w_clk_s = clk_40M_s;
   // assign fifo_r_clk_s = clk_25M_s;
   
   // DCM modules
   clk_dcm clk_dcm(
		   .U1_CLKIN_IN(CLK_50M),
		   .U1_USER_RST_IN(RST_DLL),
		   // .U1_CLKDV_OUT(clk_25M_s),
		   .U1_CLKFX_OUT(clk_40M_s),
                   .U1_CLKIN_IBUFG_OUT(clk_50M_b),
            	   .U1_CLK0_OUT(clk_50M_s),
      		   .U2_CLKFX_OUT(clk_19M_s),
      		   // .U2_CLK0_OUT(),
		   .U2_LOCKED_OUT(locked_s)
		   );

   clk_dcm2 clk_dcm2(
		     .CLKIN_IN(clk_40M_s),
		     .RST_IN(RST_DLL),
		     .CLKFX_OUT(clk_25M_s),
		     .CLKDV_OUT(clk_20M_s),
		     // .CLK0_OUT(),
		     .CLK90_OUT(clk_40M_90_s),
		     .CLK180_OUT(clk_40M_180_s),
		     .CLK270_OUT(clk_40M_270_s),
		     .LOCKED_OUT(locked2_s)
		     );

   dcm_odelay dcm_odelay(
			 .CLKIN_IN(clk_50M_b),
			 .RST_IN(RST_DLL),
			 .CLKFX_OUT(ODELAY_CLK), // 200MHz
			 // .CLKIN_IBUFG_OUT(clk_50M2_s),
			 // .CLK0_OUT(),
			 .LOCKED_OUT(locked3_s)
			 );
   
   // wire 	wait_clksync1_s;
   RST_GEN RST_GEN(
		   .CLK_IN(clk_50M_b),
		   .DCM_LOCKED(locked_s),
		   .DCM_RST(dcm_rst_s)
		   // .WAIT(wait_clksync1_s)
		   );


   assign USR_TESTPOINT = {REG_CLK, locked_s, locked2_s, clk_25M_s};

   IDELAYCTRL IDELAYCTRL(
			 .REFCLK(ODELAY_CLK),
			 .RST(sys_reset_s),
			 .RDY()
			 );

   
   /* ============================================================= */
   /*                     Data Sending Method                       */
   /* ============================================================= */
   
   //-------------------------------
   // RBCP : register Access
   //-------------------------------   
   // wire [31:0] 		SIO_ADDR; 
   // wire [7:0] 		SIO_WD;
   // wire 		SIO_WE;
   wire 		SIO_RE;
   wire 		SIO_ACK;
   wire 		SIO_RV;
   wire [7:0] 		SIO_RD;
   
   SIO_SLAVE SIO_SLAVE(
		       .RSTn(sys_reset_s),        // IN : System reset
		       .FILL_ADDR(32'h0),         // IN : Filled address for narow address-width
		       // Serial I/F
		       .SCK(REG_CLK),    	  // IN : Clock
		       .SCS(REG_ACT),    	  // IN : Active
		       .SI(REG_DI),	          // OUT : Data Input
		       .SO(REG_DO),     	  // IN : Data Output
		       // REG I/F
		       // .REG_ADDR(SIO_ADDR[31:0]), // OUT : Address[31:0]
		       // .REG_WD(SIO_WD[7:0]),	  // OUT : Data[7:0]
		       // .REG_WE(SIO_WE),	          // OUT : Write enable
		       .REG_RE(SIO_RE),	          // OUT : Read enable
		       .REG_ACK(SIO_ACK),         // IN : Access acknowledge
		       .REG_RV(SIO_RV),	          // IN : Read valid
		       .REG_RD(SIO_RD[7:0])	  // IN : Read data[7:0]
		       );
   
   ///////////////////////////////
   // UDP register access
   wire [13:0] 		conf_param_s;
   SIO_REG SIO_REG(
		   .SCK      (REG_CLK),        // in  : Clock
		   .RSTn     (~sys_reset_s),    // in  : System reset
		   // Register I/F
		   .REG_ADDR (SIO_ADDR[31:0]), // in  : Address[31:0]
		   .REG_WD   (SIO_WD[7:0]),    // in  : Data[7:0]
		   .REG_WE   (SIO_WE),         // in  : Write enable
		   .REG_RE   (SIO_RE),         // in  : Read enable
		   .REG_ACK  (SIO_ACK),        // out : Access acknowledge
		   .REG_RV   (SIO_RV),         // out : Read valid
		   .REG_RD   (SIO_RD[7:0]),    // out : Read data[7:0]
		   .CONF_PARAM(conf_param_s[13:0]),
		   .SYS_VER  (`SYS_VERSION)    // in  : System version info
		   );
   
   
   //-----------------------------------------
   // State Flags and COMMANDs declaration
   //-----------------------------------------
   
   ///////////////////////////////////////////////
   // Memory switcher
   //
   wire 		command_valid_s;
   wire [3:0] 		conf_we_s;
   wire 		ctrl_we_s;

   assign command_valid_s = ~(|SIO_ADDR[31:12]) & SIO_WE;
   
   assign conf_we_s[0] = ~(|SIO_ADDR[31:16]) & (~SIO_ADDR[15] & ~SIO_ADDR[14] & ~SIO_ADDR[13] & SIO_ADDR[12]) & SIO_WE;
   assign conf_we_s[1] = ~(|SIO_ADDR[31:16]) & (~SIO_ADDR[15] & ~SIO_ADDR[14] & SIO_ADDR[13] & ~SIO_ADDR[12]) & SIO_WE;
   assign conf_we_s[2] = ~(|SIO_ADDR[31:16]) & (~SIO_ADDR[15] & SIO_ADDR[14] & ~SIO_ADDR[13] & ~SIO_ADDR[12]) & SIO_WE;
   assign conf_we_s[3] = ~(|SIO_ADDR[31:16]) & (SIO_ADDR[15] & ~SIO_ADDR[14] & ~SIO_ADDR[13] & ~SIO_ADDR[12]) & SIO_WE;

   assign ctrl_we_s = ~(|SIO_ADDR[31:17]) & SIO_ADDR[16] & SIO_WE;
   
   ///////////////////////////////////////////////
   // Register for Commands
   //
   parameter [11:0] COMMAND = 12'ha5a;

   parameter [7:0]  COM_ACTIVE = 8'hff;
   parameter [7:0]  COM_DEACTIVE = 8'h00;

   parameter [7:0]  COM_CONTROL = 8'h01;
   parameter [7:0]  COM_CONTROL_STOP = 8'h03;
   
   parameter [7:0]  COM_CONFIGURE = 8'h05;
   parameter [7:0]  COM_START = 8'h55;
   parameter [7:0]  COM_PAUSE = 8'h5a;
   parameter [7:0]  COM_RESUME = 8'ha5;
   parameter [7:0]  COM_STOP = 8'haa;
   parameter [7:0]  COM_UNCONFIGURE = 8'ha0;
   
   reg 			conf_start_r;	//
   reg 			conf_reading_r;	//
   reg 			unconfigure_r;	// 
   
   reg 			start_daq_r;	//
   reg 			pause_daq_r;	// 
   reg 			stop_daq_r;	//
   
   reg 			DAQ_Loaded_r;	// DAQLifeCycleState LOADED
   reg 			DAQ_Ready_r;	// DAQLifeCycleState CONFIGURED
   reg 			DAQ_Running_r;	// DAQLifeCycleState RUNNING
   reg 			DAQ_Paused_r;	// DAQLifeCycleState PAUSED

   ////////////////////////////////////
   // Main Active Flag
   reg 			main_active_r = 1'b0;
   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 main_active_r <= 1'b0;
      end else begin
	 if(main_active_r == 1'b0)begin
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND
	       && SIO_WD[7:0] == COM_ACTIVE)begin
	       main_active_r <= 1'b1;
	    end
	 end else if(main_active_r == 1'b1)begin
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND
	       && SIO_WD[7:0] == COM_DEACTIVE)begin
	       main_active_r <= 1'b0;
	    end
	 end // if (main_active_r == 1'b1)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)
   
   wire 	ir_main_active_40M_s;
   wire 	ir_main_active_20M_s; // To be removed
   double_ff dff_mainact0(
			  .idata(main_active_r),
			  .oclk(clk_40M_s),
			  .odata(ir_main_active_40M_s)
			  );
   // To be removed
   double_ff dff_mainact1(
			  .idata(main_active_r),
			  .oclk(clk_20M_s),
			  .odata(ir_main_active_20M_s)
			  );
   
   
   //-------------------------------
   // SVX mode operator
   //-------------------------------
   reg          bemode_in_r;
   reg 		femode_in_r;

   wire 	bemode_out_s;
   wire 	femode_out_s;
   wire 	chmode_out_s;

   wire 	init_mode_s;
   wire 	acq_mode_s;
   wire 	digit_mode_s;
   wire 	read_mode_s;
   wire 	rst_mode_s;

   wire 	running_flag_s;

   mode_changer mode_changer(
			     .OPERATE_CLK_IN(clk_40M_s),
			     .ACTIVE_IN(ir_main_active_40M_s),
			     .RESET_IN(sys_reset_s),
			     .BEMODE_IN(bemode_in_r),
			     .FEMODE_IN(femode_in_r),
			     .RUN_FLAG_IN(running_flag_s),
			     .BEMODE_OUT(bemode_out_s),
			     .FEMODE_OUT(femode_out_s),
			     .CHMODE_OUT(chmode_out_s),
			     .INIT_FLAG(init_mode_s),
			     .ACQ_FLAG(acq_mode_s),
			     .DIGIT_FLAG(digit_mode_s),
			     .READ_FLAG(read_mode_s),
			     .RST_FLAG(rst_mode_s),
			     .ERROR_OUT()
			     );
   
   
   //-------------------------------
   // SVX configuration
   //-------------------------------
   wire [3:0] 	conf_RAM_ready_s;
   wire [3:0] 	configured_s;
   wire [3:0] 	conf_calsr_s;
   wire [3:0] 	conf_prin_s;
   wire [3:0] 	conf_feclk_s;

   wire 	all_configured_s;
   assign all_configured_s
     = (configured_s[3] | ~use_tele_s[3]) & (configured_s[2] | ~use_tele_s[2])
       & (configured_s[1] | ~use_tele_s[1]) & (configured_s[0] | ~use_tele_s[0]);
   
   conf_maker conf_maker0(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(ir_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[0]),
			  .PROUT_IN(prout_s[0]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[0]),
			  .CONFIGURED_OUT(configured_s[0]),
			  .CALSR_OUT(conf_calsr_s[0]),
			  .PRIN_OUT(conf_prin_s[0]),
			  .FECLK_OUT(conf_feclk_s[0])
			  );

   conf_maker conf_maker1(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(ir_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[1]),
			  .PROUT_IN(prout_s[1]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[1]),
			  .CONFIGURED_OUT(configured_s[1]),
			  .CALSR_OUT(conf_calsr_s[1]),
			  .PRIN_OUT(conf_prin_s[1]),
			  .FECLK_OUT(conf_feclk_s[1])
			  );

   conf_maker conf_maker2(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(ir_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[2]),
			  .PROUT_IN(prout_s[2]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[2]),
			  .CONFIGURED_OUT(configured_s[2]),
			  .CALSR_OUT(conf_calsr_s[2]),
			  .PRIN_OUT(conf_prin_s[2]),
			  .FECLK_OUT(conf_feclk_s[2])
			  );

   conf_maker conf_maker3(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(ir_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[3]),
			  .PROUT_IN(prout_s[3]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[3]),
			  .CONFIGURED_OUT(configured_s[3]),
			  .CALSR_OUT(conf_calsr_s[3]),
			  .PRIN_OUT(conf_prin_s[3]),
			  .FECLK_OUT(conf_feclk_s[3])
			  );
   
   //////////////////////////////////////////////
   // Send PRout & FECLK to PC
   reg [7:0] 	conf_fifo_in_r;
   reg 		conf_fifo_re_r;
   
   wire [7:0] 	conf_fifo_out_s;
   wire 	conf_fifo_full_s;
   wire 	conf_fifo_empty_s;
   wire 	conf_fifo_valid_s;

   wire 	ir_init_mode_s;
   double_ff dff_init(
		    .idata(init_mode_s),
		    .oclk(clk_20M_s),
		    .odata(ir_init_mode_s)
		    );
   
   fifo_conf fifo_conf(
		       .rst(sys_reset_s
			    | ~USR_ACTIVE
			    | unconfigure_r),        // IN   : Reset FIFO
		       .wr_clk(clk_20M_s),           // IN   : Clock for write data 
		       .rd_clk(clk_25M_s),           // IN   : Clock for readout data
		       .din(conf_fifo_in_r[7:0]),    // IN   : Input data
		       .wr_en(ir_init_mode_s),       // IN   : Input write enable
		       .rd_en(conf_fifo_re_r),       // IN   : Output read enable
		       .dout(conf_fifo_out_s[7:0]),  // OUT  : Output data
		       .full(conf_fifo_full_s),      // OUT  : FIFO full
		       .empty(conf_fifo_empty_s),    // OUT  : FIFO empty
		       .valid(conf_fifo_valid_s)     // OUT  : Data Valid
		       );

   always @(posedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_fifo_re_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    conf_fifo_re_r <= 1'b0;
	 end else begin
	    
	    if(USR_TX_AFULL == 1'b0 && running_flag_s == 1'b0)begin
	       if(conf_fifo_empty_s == 1'b0)begin
		  conf_fifo_re_r <= 1'b1;
	       end
	    end else begin
	       conf_fifo_re_r <= 1'b0;
	    end
	    
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_25M_s or posedge sys_reset_s)

   always @(posedge clk_20M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_fifo_in_r[7:0] <= 8'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    conf_fifo_in_r[7:0] <= 8'b0;
	 end else begin
	    conf_fifo_in_r[7:0] <= {conf_feclk_s[3], prout_s[3],
				    conf_feclk_s[2], prout_s[2],
				    conf_feclk_s[1], prout_s[1],
				    conf_feclk_s[0], prout_s[0]};
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_20M_s or posedge sys_reset_s)
   
   
   //--------------------------
   // Trigger
   //--------------------------
   wire        triged_s;
   wire        pulse_trigger_s;
   reg 	       trigger_rcved_r;
   wire [31:0] trigger_cnt_s;
   reg [31:0]  trigger_cnt_r;
   
   wire        trig_waiting_s;
   wire        busy_out_s;

   trigger trigger(
		   .TRIGGER_CLK_IN(clk_50M_s),
		   .OPERATE_CLK_IN(clk_40M_s),
		   .ACTIVE_IN(USR_ACTIVE),
		   .BUSY_IN(busy_out_s),
		   .START_IN(start_daq_r),
		   .PAUSE_IN(pause_daq_r),
		   .STOP_IN(stop_daq_r),
		   .RESET_IN(sys_reset_s),
		   .NIM_TRIGGER(NIM_IN[0]),
		   .TRIGGER_RCVED_IN(trigger_rcved_r),
		   .INT_TRIG_IN(int_trig_mode_s),
		   .TRIGGER_OUT(pulse_trigger_s),
		   .TRIGGERED_FLAG(triged_s),
		   .RUNNING_FLAG(running_flag_s),
		   .EVENT_NUM(trigger_cnt_s[31:0])
		   );

   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 trigger_rcved_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    trigger_rcved_r <= 1'b0;
	 end else begin
	    
	    if(triged_s == 1'b1/*Triggerd flag*/)begin
	       if(digit_mode_s == 1'b1/*Exit Trigger Waiting Loop*/)begin
	    	  trigger_rcved_r <= 1'b1;
	       end
	    end else if(read_mode_s == 1'b1)begin
	       trigger_rcved_r <= 1'b0;
	    end

	    // if(triged_s == 1'b1/*Triggerd flag*/)begin
	    //    if(read_mode_s == 1'b1 && all_fifo_empty_s == 1'b1)begin
	    // 	  trigger_rcved_r <= 1'b1;
	    //    end
	    // end else begin
	    //    trigger_rcved_r <= 1'b0;
	    // end
	    
	 end // else: !if(USR_ACTIVE == 1'b1)
      end // if (sys_reset_s == 1'b0)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)
   
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 trigger_cnt_r[31:0] <= 32'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    trigger_cnt_r[31:0] <= 32'b0;
	 end else begin
	    trigger_cnt_r[31:0] <= trigger_cnt_s[31:0];
	 end // else: !if(USR_ACTIVE == 1'b1)
      end // if (sys_reset_s == 1'b0)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)

   
   //--------------------------
   // SVX control signals
   //--------------------------
   wire 	  ctrl_RAM_ready_s;
   wire [15:0] 	  ctrl_sig_s;

   wire 	  ir_all_configured_s;
   double_ff dff_confed(
		    .idata(all_configured_s),
		    .oclk(clk_40M_s),
		    .odata(ir_all_configured_s)
		    );
   
   wire 	  fifo_all_r_ready_s;
   wire 	  ir_fifo_all_r_ready_s;
   double_ff dff_fifo_all_r_ready(
				  .idata(fifo_all_r_ready_s),
				  .oclk(clk_40M_s),
				  .odata(ir_fifo_all_r_ready_s)
				  );

   // wire 	  mux_clock_s;
   // BUFGMUX_CTRL BUFGMUX_inst(.O(mux_clock_s), .S(ir_init_mode_s),
   // 			     .I0(clk_40M_s), .I1(clk_20M_s));
   
   ctrl_sig_maker ctrl_sig_maker(
				 .OPERATE_CLK_IN(clk_40M_s),
				 .REG_CLK_IN(REG_CLK),
				 .ACTIVE_IN(ir_main_active_40M_s),
				 .RESET_IN(sys_reset_s | unconfigure_r),
				 .SIO_ADDR_IN(SIO_ADDR[13:0]),
				 .SIO_WD_IN(SIO_WD[7:0]),
				 .SIO_WE_IN(ctrl_we_s),
				 .TRIGGER_IN(triged_s),
				 .READ_START_IN(ir_all_configured_s),
				 .READ_M_END_IN(ir_fifo_all_r_ready_s),
				 .CONF_PARAM_IN({6'd6, 8'd41}/*conf_param_s[13:0]*/),
				 .RAM_LOADED_OUT(ctrl_RAM_ready_s),
				 .TRIG_WAIT_OUT(trig_waiting_s),
				 .CTRL_SIG_OUT(ctrl_sig_s[15:0])
				 );

   // 141208 Yajima : This section added
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 bemode_in_r <= 1'b1;
	 femode_in_r <= 1'b0;
      end else begin
	 if(ir_main_active_40M_s == 1'b0)begin
	    bemode_in_r <= 1'b1;
	    femode_in_r <= 1'b0;
	 end else begin

	    if(ir_all_configured_s == 1'b1)begin
	       if(ctrl_sig_s[15]/*CHMODE*/ == 1'b1)begin
		  bemode_in_r <= ctrl_sig_s[13];
		  femode_in_r <= ctrl_sig_s[14];
	       end
	    end else if(conf_start_r == 1'b1)begin
	       bemode_in_r <= 1'b0;
	       femode_in_r <= 1'b0;
	    end

	 end // else: !if(ir_main_active_40M_s == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)

   wire [7:0] ctrl_sig_inv_s;
   assign ctrl_sig_inv_s[7:0] = {ctrl_sig_s[0], ctrl_sig_s[1],
				 ctrl_sig_s[2], ctrl_sig_s[3],
				 ctrl_sig_s[4], ctrl_sig_s[5],
				 ctrl_sig_s[6], ctrl_sig_s[7]};

   wire       ir0_conf_calsr_s;
   wire       ir1_conf_calsr_s;
   wire       ir2_conf_calsr_s;
   wire       ir3_conf_calsr_s;
   double_ff dff_calsr0(.idata(conf_calsr_s[0]),
		    .oclk(clk_40M_s), .odata(ir0_conf_calsr_s)
		    );
   double_ff dff_calsr1(.idata(conf_calsr_s[1]),
			.oclk(clk_40M_s), .odata(ir1_conf_calsr_s)
			);
   double_ff dff_calsr2(.idata(conf_calsr_s[2]),
			.oclk(clk_40M_s), .odata(ir2_conf_calsr_s)
			);
   double_ff dff_calsr3(.idata(conf_calsr_s[3]),
			.oclk(clk_40M_s), .odata(ir3_conf_calsr_s)
			);
   
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 bus_r[31:0] <= 32'b0011_0011_0011_0011_0011_0011_0011_0011;
      end else begin
	 if(ir_main_active_40M_s == 1'b0)begin
	    bus_r[31:0] <= 32'b0011_0011_0011_0011_0011_0011_0011_0011;	    
	 end else begin
	    bus_r[31:0] <= (init_mode_s == 1'b1) ? {ir3_conf_calsr_s, 7'b011_0011,
						    ir2_conf_calsr_s, 7'b011_0011,
						    ir1_conf_calsr_s, 7'b011_0011,
						    ir0_conf_calsr_s, 7'b011_0011}
			   : { 4{ctrl_sig_inv_s[7:0]} };
	 end // else: !if(ir_main_active_40M_s == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)

   reg 	    beclk_r;
   reg      feclk_r;
   reg 	    clk_sel_r;
   reg 	    prin_r;
   
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 beclk_r <= 1'b0;
	 feclk_r <= 1'b0;
	 clk_sel_r <= 1'b0;
	 prin_r <= 1'b0;
      end else begin
	 if(ir_main_active_40M_s == 1'b0)begin
	    beclk_r <= 1'b0;
	    feclk_r <= 1'b0;
	    clk_sel_r <= 1'b0;
	    prin_r <= 1'b0;
	 end else begin
	    beclk_r <= ctrl_sig_s[9];
	    feclk_r <= ctrl_sig_s[10];
	    clk_sel_r <= ctrl_sig_s[11];
	    prin_r <= ctrl_sig_s[12];
	 end // else: !if(ir_main_active_40M_s == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)


   always @(posedge clk_40M_s)begin
      feclk2_r[3:0] = (init_mode_s == 1'b1 || conf_start_r == 1'b1) ? ~conf_feclk_s[3:0] : { 4{~feclk_r} };
      prin2_r[3:0] = (init_mode_s == 1'b1) ? ~conf_prin_s[3:0] : { 4{~prin_r} };
   end
   
   /* ============================================================= */
   /*                     Data Receiving Method                     */
   /* ============================================================= */
   
   //---------------------------------
   // Fifo
   //---------------------------------
   wire [31:0] fifo_out_s;
   wire [3:0]  fifo_full_s;
   wire [3:0]  fifo_empty_s;
   wire [3:0]  fifo_dvalid_s;
   wire [3:0]  fifo_r_ready_s;
   reg [3:0]   fifo_re_r;

   reg [7:0]   event_header_r;
   reg 	       evheader_writing_r;

   reg ir_evheader_writing_r;
   always @(negedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 ir_evheader_writing_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    ir_evheader_writing_r <= 1'b0;
	 end else begin
	    ir_evheader_writing_r <= evheader_writing_r;
	 end
      end
   end // always @ (negedge clk_40M_s or posedge sys_reset_s)

   wire [31:0] bus_data_s;

   wire [7:0]  fifo_mod0_in_s;
   assign fifo_mod0_in_s[7:0] = (read_mode_s == 1'b1) ?
				bus_data_s[7:0] : event_header_r[7:0];
   fifo_mod fifo_mod0(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(fifo_mod0_in_s[7:0]),
		      .READ_MODE_IN(read_mode_s),
		      .FIFO_RE_IN(fifo_re_r[0]),
		      .OBDV_IN(obdv_s[0]),
		      .FIFO_FORCE_WE(ir_evheader_writing_r),
		      .OUTPUT_DATA(fifo_out_s[7:0]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[0]),
		      .FIFO_FULL_OUT(fifo_full_s[0]),
		      .DATA_VALID_OUT(fifo_dvalid_s[0]),
		      .FIFO_READ_READY(fifo_r_ready_s[0])
		      );

   fifo_mod fifo_mod1(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(bus_data_s[15:8]),
		      .READ_MODE_IN(read_mode_s),
		      .FIFO_RE_IN(fifo_re_r[1]),
		      .OBDV_IN(obdv_s[1]),
		      .FIFO_FORCE_WE(1'b0),
		      .OUTPUT_DATA(fifo_out_s[15:8]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[1]),
		      .FIFO_FULL_OUT(fifo_full_s[1]),
		      .DATA_VALID_OUT(fifo_dvalid_s[1]),
		      .FIFO_READ_READY(fifo_r_ready_s[1])
		      );
   
   fifo_mod fifo_mod2(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(bus_data_s[23:16]),
		      .READ_MODE_IN(read_mode_s),
		      .FIFO_RE_IN(fifo_re_r[2]),
		      .OBDV_IN(obdv_s[2]),
		      .FIFO_FORCE_WE(1'b0),
		      .OUTPUT_DATA(fifo_out_s[23:16]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[2]),
		      .FIFO_FULL_OUT(fifo_full_s[2]),
		      .DATA_VALID_OUT(fifo_dvalid_s[2]),
		      .FIFO_READ_READY(fifo_r_ready_s[2])
		      );
   
   fifo_mod fifo_mod3(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(bus_data_s[31:24]),
		      .READ_MODE_IN(read_mode_s),
		      .FIFO_RE_IN(fifo_re_r[3]),
		      .OBDV_IN(obdv_s[3]),
		      .FIFO_FORCE_WE(1'b0),
		      .OUTPUT_DATA(fifo_out_s[31:24]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[3]),
		      .FIFO_FULL_OUT(fifo_full_s[3]),
		      .DATA_VALID_OUT(fifo_dvalid_s[3]),
		      .FIFO_READ_READY(fifo_r_ready_s[3])
		      );
   
   wire        fifo_all_read_s;
   assign fifo_all_r_ready_s
     = (fifo_r_ready_s[3] | ~use_tele_s[3]) & (fifo_r_ready_s[2] | ~use_tele_s[2])
       & (fifo_r_ready_s[1] | ~use_tele_s[1]) & (fifo_r_ready_s[0] | ~use_tele_s[0]);
   assign fifo_all_read_s
     = ~((fifo_r_ready_s[3] & use_tele_s[3]) | (fifo_r_ready_s[2] & use_tele_s[2])
	 | (fifo_r_ready_s[1] & use_tele_s[1]) | (fifo_r_ready_s[0] & use_tele_s[0]));
   
   reg 	       data_reading_r;
   always @(posedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 fifo_re_r[3:0] <= 4'b0000;
	 data_reading_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    fifo_re_r[3:0] <= 4'b0000;
	    data_reading_r <= 1'b0;
	 end else begin
	    
	    if(USR_TX_AFULL == 1'b0 && evheader_writing_r == 1'b0)begin
	       if(fifo_empty_s[0] == 1'b0
		  && use_tele_s[0] == 1'b1 && fifo_all_r_ready_s == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0001;
		  data_reading_r <= 1'b1;
	       end else if(fifo_empty_s[1] == 1'b0
			   && data_reading_r == 1'b1 && use_tele_s[1] == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0010;
	       end else if(fifo_empty_s[2] == 1'b0
			   && data_reading_r == 1'b1 && use_tele_s[2] == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0100;
	       end else if(fifo_empty_s[3] == 1'b0
			   && data_reading_r == 1'b1 && use_tele_s[3] == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b1000;
	       end else if(fifo_all_read_s == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0000;
		  data_reading_r <= 1'b0;
	       end else begin
		  fifo_re_r[3:0] <= 4'b0000;
	       end
	    end else begin
	       fifo_re_r[3:0] <= 4'b0000;
	    end // else: !if(USR_TX_AFULL == 1'b0 && evheader_writing_r == 1'b0)
	    
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_25M_s or posedge sys_reset_s)
   
   
   //---------------------------------
   // Timestamp
   //---------------------------------
   wire        ts_clk_s;
   reg [23:0]  time_stamp_r;
   wire [23:0] time_stamp_s;
   
   assign ts_clk_s = ~NIM_IN[1];
   // assign ts_clk_s = clk_100k_s;

   time_stamp time_stamp(
			 .CLK(clk_40M_s),
			 .TS_CLK(ts_clk_s),
			 .TRIGGER_IN(triged_s),
			 .RESET(stop_daq_r | ~main_active_r),
			 .BUSY(busy_out_s),
			 .TIMESTAMP_OUT(time_stamp_s[23:0])
			 );

   always@(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 time_stamp_r[23:0] <= 24'd0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    time_stamp_r[23:0] <= 24'd0;
	 end else begin
	    time_stamp_r[23:0] <= time_stamp_s[23:0];	 
	 end
      end
   end // always@ (posedge clk_40M_s or posedge sys_reset_s)

   
   //---------------------------------
   // Event Header
   //---------------------------------
   wire        header_we_s;
   wire [7:0]  event_header_s;
   
   header_tele header(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .ACTIVE(USR_ACTIVE),
		      .EVENT_NUM_IN(trigger_cnt_r[31:0]),
		      .TIMESTAMP_IN(time_stamp_r[23:0]),
		      .TRIGGER_IN(pulse_trigger_s),
		      .TRIGGER_RCVED_IN(trigger_rcved_r),
		      .RESET_IN(sys_reset_s),
		      .HEADER_OUT(event_header_s[7:0]),
		      .HEADER_WE(header_we_s)
		      );

   always@(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 event_header_r[7:0] <= 8'b0;
	 evheader_writing_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    event_header_r[7:0] <= 8'b0;
	    evheader_writing_r <= 1'b0;
	 end else begin
	    event_header_r[7:0] <= event_header_s[7:0];
	    evheader_writing_r <= header_we_s;
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always@ (posedge clk_40M_s or posedge sys_reset_s)
   

   //---------------------------------
   // Signal assignments
   //---------------------------------
   ///////////////////////////////////////////
   // BUSY
   wire    all_fifo_empty_s;
   
   assign all_fifo_empty_s
     = (fifo_empty_s[3] | ~use_tele_s[3]) & (fifo_empty_s[2] | ~use_tele_s[2])
       & (fifo_empty_s[1] | ~use_tele_s[1]) & (fifo_empty_s[0] | ~use_tele_s[0]);
   assign busy_out_s = ~(trig_waiting_s & ~triged_s
			 & running_flag_s & all_fifo_empty_s);
   

   ///////////////////////////////////////////
   // Signals
   // assign BEMODE[3:0] = { 4{bemode_out_s} };
   // assign FEMODE[3:0] = { 4{femode_out_s} };
   // assign CHMODE[3:0] = { 4{chmode_out_s} };
   
   // assign beclk_s = (clk_sel_r == 1'b1) ? clk_40M_s : beclk_r;

   assign DDATA[3:0] = (read_mode_s == 1'b1) ? 4'b1111 : 4'b0000;
   assign RDATA[3:0] = (read_mode_s == 1'b0) ? 4'b1111 : 4'b0000;
   
   ///////////////////////////////////////////
   // BUS Lines
   wire [31:0] bus_s;
   assign bus_s[31:0] = ~bus_r[31:0];

   assign PARST[3:0] =   (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[28], ~bus_s[20], ~bus_s[12], ~bus_s[4]};
   assign PIPERD1[3:0] = (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[30], ~bus_s[22], ~bus_s[14], ~bus_s[6]};
   assign PIPERD2[3:0] = (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[26], ~bus_s[18], ~bus_s[10], ~bus_s[2]};
   assign L1A[3:0] =     (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[29], ~bus_s[21], ~bus_s[13], ~bus_s[5]};
   assign CALSR[3:0] =   (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[31], ~bus_s[23], ~bus_s[15], ~bus_s[7]};

   assign LED[7:0] = (DIP_SW_s[4] == 1'b1) ? {acq_mode_s, bus_r[6:0]} : 8'b0000_0000;
   // assign LED[7:0] = (DIP_SW_s[4] == 1'b1) ? {init_mode_s, acq_mode_s, digit_mode_s, read_mode_s, 
   // 					      command_valid_s, configured_s[0], conf_start_r, conf_RAM_ready_s[0]}
   // 		     : 8'b0000_0000;

   //////////////////////////////////////////
   // NIM Output
   assign NIM_OUT[0] = ~busy_out_s;
   assign NIM_OUT[1] = prout_s[0];

   //////////////////////////////////////////
   // define BUS I/O selector
   wire        io_state_sel_s;
   assign io_state_sel_s = read_mode_s;
   
   
   //----------------------------------------------------
   //  TC sender : Send data to PC with SiTCP method
   //----------------------------------------------------
   
   reg [7:0]   TX_WD_r;

   function [7:0] TX_Data_sw_f;
      input [5:0] TX_Data_sw;
      case(TX_Data_sw)
	6'b100000: TX_Data_sw_f = conf_fifo_out_s[7:0];
	// 6'b010000: TX_Data_sw_f = event_header_r[7:0];
	6'b001000: TX_Data_sw_f = fifo_out_s[31:24];
	6'b000100: TX_Data_sw_f = fifo_out_s[23:16];
	6'b000010: TX_Data_sw_f = fifo_out_s[15:8];
	6'b000001: TX_Data_sw_f = fifo_out_s[7:0];
	default: TX_Data_sw_f = 8'd0;
      endcase // case (TX_Data_sw)
   endfunction // case

   always @(negedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 TX_WD_r[7:0] <= 8'd0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    TX_WD_r[7:0] <= 8'd0;
	 end else begin
	    TX_WD_r <= TX_Data_sw_f({conf_fifo_valid_s,
				     evheader_writing_r, fifo_dvalid_s[3:0]});
	 end
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (negedge clk_25M_s or posedge sys_reset_s)

   reg ir_conf_fifo_valid_r;
   // reg ir_evheader_writing_r;
   reg ir_fifo_dvalid_r;
   
   always @(negedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 ir_conf_fifo_valid_r <= 1'b0;
	 // ir_evheader_writing_r <= 1'b0;
	 ir_fifo_dvalid_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    ir_conf_fifo_valid_r <= 1'b0;
	    // ir_evheader_writing_r <= 1'b0;
	    ir_fifo_dvalid_r <= 1'b0;
	 end else begin
	    ir_conf_fifo_valid_r <= conf_fifo_valid_s;
	    // ir_evheader_writing_r <= evheader_writing_r;
	    ir_fifo_dvalid_r <= |fifo_dvalid_s;
	 end
      end
   end
   
   assign USR_CLK = clk_25M_s;
   assign USR_TX_WD[7:0] = TX_WD_r[7:0];
   assign USR_TX_WE = ir_fifo_dvalid_r |
		      ir_conf_fifo_valid_r;
   // | ir_evheader_writing_r;

   assign USR_CLOSE_ACK = USR_CLOSE_REQ;


   //---------------------------------
   // State Flags and COMMANDs
   //---------------------------------
   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_reading_r <= 1'b0;	 
      end else begin
	 if(main_active_r == 1'b0)begin
	    conf_reading_r <= 1'b0;
	 end else begin
	    
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND)begin
	       if(SIO_WD[7:0] == COM_CONTROL_STOP)begin
		  conf_reading_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_CONTROL
			   || SIO_WD[7:0] == COM_UNCONFIGURE)begin
		  conf_reading_r <= 1'b1;
	       end
	    end
	    
	 end // else: !if(main_active_r == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)

   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_start_r <= 1'b0;
      end else begin
	 if(main_active_r == 1'b0)begin
	    conf_start_r <= 1'b0;
	 end else begin
	    
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND
	       && SIO_WD[7:0] == COM_CONFIGURE)begin
	       conf_start_r <= 1'b1;
	    end else begin
	       conf_start_r <= 1'b0;
	    end
	    
	 end // else: !if(main_active_r == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)

   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 start_daq_r <= 1'b0;
	 stop_daq_r <= 1'b0;
	 pause_daq_r <= 1'b0;
	 unconfigure_r <= 1'b0;
      end else begin
	 if(main_active_r == 1'b0)begin
	    start_daq_r <= 1'b0;
	    stop_daq_r <= 1'b0;
	    pause_daq_r <= 1'b0;
	    unconfigure_r <= 1'b0;
	 end else begin
	    
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND)begin
	       if(SIO_WD[7:0] == COM_START)begin
		  start_daq_r <= 1'b1;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_STOP)begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b1;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_PAUSE)begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b1;
		  unconfigure_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_UNCONFIGURE)begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b1;
	       end else begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b0;
	       end
	    end
	    
	 end // else: !if(main_active_r == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)

   
   //---------------------------------
   // Prevent to timing fluctuation
   //---------------------------------
   wire [3:0] ir_beclk_s;

   assign beclk1_s = (clk_sel_r == 1'b1) ? 1'b1 : ~beclk_r;
   assign beclk2_s = (clk_sel_r == 1'b1) ? 1'b0 : ~beclk_r;
   
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk1(.Q(ir_beclk_s[0]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk2(.Q(ir_beclk_s[1]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk3(.Q(ir_beclk_s[2]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk4(.Q(ir_beclk_s[3]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );

   (* IOB = "FORCE" *) reg [3:0] ir_feclk_r;
   (* IOB = "FORCE" *) reg [3:0] ir_prin_r;
   always @(posedge clk_40M_s)begin
      ir_feclk_r[3:0] <= feclk2_r[3:0];
      ir_prin_r[3:0] <= prin2_r[3:0];
   end
   
   (* IOB = "FORCE" *) reg [31:0] ir_bus_r;
   (* IOB = "FORCE" *) reg [3:0]  ir_chmode_r;
   (* IOB = "FORCE" *) reg [3:0]  ir_bemode_r;
   (* IOB = "FORCE" *) reg [3:0]  ir_femode_r;
   always @(posedge clk_40M_s)begin
      ir_bus_r[31:0] <= bus_s[31:0];
      ir_chmode_r[3:0] <= { 4{chmode_out_s} };
      ir_bemode_r[3:0] <= { 4{bemode_out_s} };
      ir_femode_r[3:0] <= { 4{femode_out_s} };
   end

   // CLK Delay
   wire [3:0] id_beclk_s;
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK0(.DATAOUT(id_beclk_s[0]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_beclk_s[0]), .RST(1'b0), .T(1'b0));
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK1(.DATAOUT(id_beclk_s[1]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_beclk_s[1]), .RST(1'b0), .T(1'b0));
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK2(.DATAOUT(id_beclk_s[2]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_beclk_s[2]), .RST(1'b0), .T(1'b0));
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK3(.DATAOUT(id_beclk_s[3]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_beclk_s[3]), .RST(1'b0), .T(1'b0));
   
   // BUS line IODELAY
   // wire [31:0] id_bus_s;
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[0]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[0]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[1]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[1]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[2]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[2]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[3]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[3]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[4]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[4]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[5]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[5]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[6]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[6]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[7]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[7]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[8]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[8]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[9]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(ir_bus_r[9]), .RST(sys_reset_s), .T(1'b0));
   
   //telescope1
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS0_MAP(.O(bus_data_s[0]), .IO(BUS[0]), .I(ir_bus_r[0]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS1_MAP(.O(bus_data_s[1]), .IO(BUS[1]), .I(ir_bus_r[1]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS2_MAP(.O(bus_data_s[2]), .IO(BUS[2]), .I(ir_bus_r[2]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS3_MAP(.O(bus_data_s[3]), .IO(BUS[3]), .I(ir_bus_r[3]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS4_MAP(.O(bus_data_s[4]), .IO(BUS[4]), .I(ir_bus_r[4]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS5_MAP(.O(bus_data_s[5]), .IO(BUS[5]), .I(ir_bus_r[5]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS6_MAP(.O(bus_data_s[6]), .IO(BUS[6]), .I(ir_bus_r[6]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS7_MAP(.O(bus_data_s[7]), .IO(BUS[7]), .I(ir_bus_r[7]), .T(io_state_sel_s));
   
   //telescope2
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS8_MAP(.O(bus_data_s[8]), .IO(BUS[8]), .I(ir_bus_r[8]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS9_MAP(.O(bus_data_s[9]), .IO(BUS[9]), .I(ir_bus_r[9]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS10_MAP(.O(bus_data_s[10]), .IO(BUS[10]), .I(ir_bus_r[10]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS11_MAP(.O(bus_data_s[11]), .IO(BUS[11]), .I(ir_bus_r[11]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS12_MAP(.O(bus_data_s[12]), .IO(BUS[12]), .I(ir_bus_r[12]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS13_MAP(.O(bus_data_s[13]), .IO(BUS[13]), .I(ir_bus_r[13]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS14_MAP(.O(bus_data_s[14]), .IO(BUS[14]), .I(ir_bus_r[14]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS15_MAP(.O(bus_data_s[15]), .IO(BUS[15]), .I(ir_bus_r[15]), .T(io_state_sel_s));
   
   //telescope3
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS16_MAP(.O(bus_data_s[16]), .IO(BUS[16]), .I(ir_bus_r[16]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS17_MAP(.O(bus_data_s[17]), .IO(BUS[17]), .I(ir_bus_r[17]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS18_MAP(.O(bus_data_s[18]), .IO(BUS[18]), .I(ir_bus_r[18]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS19_MAP(.O(bus_data_s[19]), .IO(BUS[19]), .I(ir_bus_r[19]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS20_MAP(.O(bus_data_s[20]), .IO(BUS[20]), .I(ir_bus_r[20]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS21_MAP(.O(bus_data_s[21]), .IO(BUS[21]), .I(ir_bus_r[21]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS22_MAP(.O(bus_data_s[22]), .IO(BUS[22]), .I(ir_bus_r[22]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS23_MAP(.O(bus_data_s[23]), .IO(BUS[23]), .I(ir_bus_r[23]), .T(io_state_sel_s));
   
   //telescope4
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS24_MAP(.O(bus_data_s[24]), .IO(BUS[24]), .I(ir_bus_r[24]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS25_MAP(.O(bus_data_s[25]), .IO(BUS[25]), .I(ir_bus_r[25]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS26_MAP(.O(bus_data_s[26]), .IO(BUS[26]), .I(ir_bus_r[26]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS27_MAP(.O(bus_data_s[27]), .IO(BUS[27]), .I(ir_bus_r[27]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS28_MAP(.O(bus_data_s[28]), .IO(BUS[28]), .I(ir_bus_r[28]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS29_MAP(.O(bus_data_s[29]), .IO(BUS[29]), .I(ir_bus_r[29]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS30_MAP(.O(bus_data_s[30]), .IO(BUS[30]), .I(ir_bus_r[30]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS31_MAP(.O(bus_data_s[31]), .IO(BUS[31]), .I(ir_bus_r[31]), .T(io_state_sel_s));

   //---------------------------------------------------------//
   //telescope1
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE1_MAP(.O(CHMODE[0]), .I(ir_chmode_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE1_MAP(.O(BEMODE[0]), .I(ir_bemode_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE1_MAP(.O(FEMODE[0]), .I(ir_femode_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK1_MAP(.O(FECLK[0]), .I(ir_feclk_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK1_MAP(.O(BECLK[0]), .I(id_beclk_s[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin1_MAP(.O(PRin[0]), .I(ir_prin_r[0]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV1_MAP(.O(obdv_s[0]), .I(OBDV[0]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout1_MAP(.O(prout_s[0]), .I(PRout[0]));
   
   //telescope2
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE2_MAP(.O(CHMODE[1]), .I(ir_chmode_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE2_MAP(.O(BEMODE[1]), .I(ir_bemode_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE2_MAP(.O(FEMODE[1]), .I(ir_femode_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK2_MAP(.O(FECLK[1]), .I(ir_feclk_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK2_MAP(.O(BECLK[1]), .I(id_beclk_s[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin2_MAP(.O(PRin[1]), .I(ir_prin_r[1]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV2_MAP(.O(obdv_s[1]), .I(OBDV[1]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout2_MAP(.O(prout_s[1]), .I(PRout[1]));
   
   //telescope3
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE3_MAP(.O(CHMODE[2]), .I(ir_chmode_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE3_MAP(.O(BEMODE[2]), .I(ir_bemode_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE3_MAP(.O(FEMODE[2]), .I(ir_femode_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK3_MAP(.O(FECLK[2]), .I(ir_feclk_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK3_MAP(.O(BECLK[2]), .I(id_beclk_s[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin3_MAP(.O(PRin[2]), .I(ir_prin_r[2]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV3_MAP(.O(obdv_s[2]), .I(OBDV[2]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout3_MAP(.O(prout_s[2]), .I(PRout[2]));
   
   //telescope4
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE4_MAP(.O(CHMODE[3]), .I(ir_chmode_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE4_MAP(.O(BEMODE[3]), .I(ir_bemode_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE4_MAP(.O(FEMODE[3]), .I(ir_femode_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK4_MAP(.O(FECLK[3]), .I(ir_feclk_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK4_MAP(.O(BECLK[3]), .I(id_beclk_s[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin4_MAP(.O(PRin[3]), .I(ir_prin_r[3]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV4_MAP(.O(obdv_s[3]), .I(OBDV[3]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout4_MAP(.O(prout_s[3]), .I(PRout[3]));
   
endmodule
