#include "daq.h"

#include "stdio.h"
#include "stdlib.h"
#include "unistd.h"


#define SVX4_MAGIC_NUM 25104 //magic number

namespace svx{

  Daq::Daq(std::string file_name){
    rawfile_name = file_name;
    msg_len = 0;
    msg_len_tot = 0;
    first_event = true;
    dummy_data = true;
    verbose_level = g_verbose_level;
  }


  Daq::Daq(Comm* arg_comm, std::string file_name){
    comm = arg_comm;
  
    rawfile_name = file_name;
    msg_len = 0;
    msg_len_tot = 0;
    first_event = true;
    dummy_data = false;
    verbose_level = g_verbose_level;
  }


  Daq::~Daq(){

  }


  int Daq::read_svxdata(){
  
    for(int i=0; i<STORAGE_DATASIZE; i++){
      if(first_event)
	store_data[i] = 0x00;
      else{
	if( (msg_len-1)-i >= 0 )
	  store_data[i] = raw_data[(msg_len-1)-i];
      }
    }

    int select_re;
    select_re = comm->read_tcp_data(msg_len, raw_data, 100000);    

    write_svxdata(rawfile_name);
    msg_len_tot += msg_len;
    // std::cout << msg_len_tot << std::endl;
  
    return select_re;
  }


  int Daq::read_bitdata(std::string file_name){
  
    static bool read_all_data = false;
    static std::ios::pos_type read_pos;
    static int buf_count = 0;
    static int byte_count = -1;
    static int max_data_length = -1;
    if(first_event){
      read_all_data = false;
      read_pos = 0;
      buf_count = -1;
      byte_count = -1;
      max_data_length = -1;
    }

    if(read_all_data){
      std::cout << "Daq::read_bitdata() : Read All Dummy Data completly." << std::endl;
      throw (char*)"Read All Dummy Data completly.";
    }

    for(int i=0; i<STORAGE_DATASIZE; i++){
      if(first_event)
	store_data[i] = 0x00;
      else{
	if( (msg_len-1)-i >= 0 )
	  store_data[i] = raw_data[(msg_len-1)-i];
      }
    }

    std::ifstream ifs(file_name.c_str());
    if(!ifs){
      std::cout << "Error! Failed to Open File. (" << file_name <<')' << std::endl;
      exit(-1);
    }
  
    std::string tmp_byte;
    std::string tmp_data;

    std::cout << "Reading test dummy data..." << std::endl;
  
    if(first_event){
      ifs >> tmp_byte;
      byte_count = atoi(tmp_byte.c_str());
    }else
      ifs.seekg(read_pos);

    int data_length = 0;
    while(true){
      ifs >> tmp_data;
      raw_data[byte_count] = (unsigned char)atoi(tmp_data.c_str());;
      // std::cout << byte_count << " " << (int)raw_data[byte_count] << std::endl;
      ifs >> tmp_byte;
      byte_count = atoi(tmp_byte.c_str());
    
      data_length++;
    
      if(ifs.eof()){
	std::cout << "Reach The End of The File" << std::endl;
	read_all_data = true;
	break;
      }else if(byte_count == 0){
	read_pos = ifs.tellg();
	break;
      }
    }

    msg_len = data_length;
    ifs.close();

    if(max_data_length < data_length)
      max_data_length = data_length;
  
    std::cout << "buf_count  = " << buf_count
	      << ", Data Length = " << data_length << std::endl;
    std::cout << "Max Data Length = " << max_data_length << std::endl; // For Debug
  
    buf_count++;
  
    return 0;
  }


  int Daq::write_svxdata(std::string rawfile_name){

    std::ios::openmode open_flag = std::ios::app;
    
    // write to text file
    if(first_event){
      open_flag = std::ios::trunc;
    }else
      open_flag = std::ios::app;
		
    std::ofstream outf(rawfile_name.c_str(), open_flag);
    if(!outf){
      std::cout << "Error! Failed to Open File. (" << rawfile_name <<')' << std::endl;
      exit(1); // return -1;
    }

    for(int i=0; i<msg_len; i++)
      outf << i << '\t' << (int)(raw_data[i]) << std::endl;
  
    outf.close();
  
    return 0;
  }


  int Daq::write_errdata(std::string errmsg, int data_cnt, int backward, int forward){
    std::cout << data_cnt << " Daq::write_errdata() : " << errmsg << std::endl;

    std::string errfile_name = rawfile_name;
    errfile_name += ".error";
    static bool first_write = true;
    static std::string filename = errfile_name;
    std::ios::openmode open_flag = std::ios::app;
    
    // write to text file
    if(first_write
       || (filename.compare(errfile_name) != 0)){
      first_write = false;
      open_flag = std::ios::trunc;
    }else
      open_flag = std::ios::app;
		
    std::ofstream outf(errfile_name.c_str(), open_flag);
    if(!outf){
      std::cout << "Error! Failed to Open File. (" << errfile_name <<')' << std::endl;
      exit(1); // return -1;
    }

    int begin = -1*STORAGE_DATASIZE;
    int end = msg_len;
    if((data_cnt - backward) >= begin)   begin = data_cnt - backward;
    if((data_cnt + forward) < msg_len)   end  = data_cnt + forward;

    outf << "ERROR : " << errmsg << std::endl;
    outf << "Event_Number: " << ev_num << std::endl;
    for(int i=begin; i<end; i++){
      if(i==data_cnt)  outf << i << '\t' << (int)(get_rawdata(i)) << "\t*" << std::endl;
      else             outf << i << '\t' << (int)(get_rawdata(i)) << std::endl;
    }
  
    outf.close();
  
    return 0;
  }


  int Daq::GetDecodedData(char* Data){

    SVX_EF *send_ef;
    // SVX_EF_HEADER ef_header;

    int data_size_sum = 0;
    char *send_buffer;
    char *buf_pointer;

    m_ef_header.magicdata = SVX4_MAGIC_NUM;
    m_ef_header.ef_number = TELESCOPE_NUM;
  
    // Get Total Data_size
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      m_ef_header.ef_length[tele_i]
	= sizeof(struct SVX_EF)
	+ sizeof(struct SVXdata)*((int)m_svxdata_buf[tele_i].svx_data.size() - 1);
      data_size_sum += m_ef_header.ef_length[tele_i];
    }
  
    m_ef_header.totalsize = data_size_sum + sizeof(struct SVX_EF_HEADER);
#ifdef NEW_HEADER_FORMAT
    // m_ef_header.data_quality = m_svxdata_buf[0].status;
    m_ef_header.data_type = 0;
#endif

    //Memory Allocation
    send_buffer = (char*)malloc(m_ef_header.totalsize);

    //Check Allocated Memory
    if(send_buffer == NULL){
      std::cout << "Daq::GetDecodedData() : Memory Allocate Error(send_buffer)" << std::endl;
      return -1;
    }
  
    //copy Header
    memcpy(send_buffer, (char*)&m_ef_header, sizeof(struct SVX_EF_HEADER));
    buf_pointer = send_buffer + sizeof(struct SVX_EF_HEADER);

    //copy M_Svxdata_Buf data to SVX_EF
    int data_size = 0;
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      //Memory Allocation
      data_size = m_ef_header.ef_length[tele_i];
      send_ef = (SVX_EF*)malloc(data_size);
    
      //Check Allocated Memory
      if(send_ef == NULL){
	std::cout << "Daq::GetDecodedData() : Memory Allocate Error(send_ef)" << std::endl;
	return -1;
      }
    
      send_ef->telescope_No = m_svxdata_buf[tele_i].telescope_No;
    
      for(int chip_i=0; chip_i<4; chip_i++){
	send_ef->chipID[chip_i] = m_svxdata_buf[tele_i].chipID[chip_i];
	send_ef->pipelineID[chip_i] = m_svxdata_buf[tele_i].pipelineID[chip_i];
      }
    
      send_ef->hitnum = m_svxdata_buf[tele_i].hitnum;
      copy(m_svxdata_buf[tele_i].svx_data.begin(), m_svxdata_buf[tele_i].svx_data.end(), send_ef->svx_data);
    
      memcpy(buf_pointer, send_ef, data_size);
      buf_pointer += data_size;
      free(send_ef);
    }
    
    if(verbose_level > 0){
      std::cout << "Daq::GetDecodedData() : Total Data Size == " << m_ef_header.totalsize
		<< " (Header Size == " << sizeof(struct SVX_EF_HEADER)
		<< ')' << std::endl;
    }

    memcpy(Data, send_buffer, m_ef_header.totalsize);
    free(send_buffer);
  
    return 0;
  }


  int Daq::GetDecodedDataSize(){
  
    int ef_length = 0;
    int data_size_sum = 0;
    int totalsize = -1;

    // Get Total Data_size
    for(int tele_i=0; tele_i < TELESCOPE_NUM; tele_i++){
      ef_length
	= sizeof(struct SVX_EF) - sizeof(struct SVXdata)
	+ sizeof(struct SVXdata)*(int)m_svxdata_buf[tele_i].svx_data.size();
      data_size_sum += ef_length;
    }
  
    totalsize = data_size_sum + sizeof(struct SVX_EF_HEADER);

    return totalsize;
  }


  ///////////////////////////////////////////////////////////////
  //  DECODER
  ///////////////////////////////////////////////////////////////
  int Daq::get_config_param(SVXconfig* config){
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      for(int chip_i=0; chip_i<4; chip_i++){
	chipID_config[tele_i][chip_i] = (int)config->getparam(ID, tele_i, chip_i);
    
	if((int)chipID_config[tele_i][chip_i] < 128){
	  std::cout << "chipID has invalid value. ("
		    << (int)chipID_config[tele_i][chip_i] << ')' << std::endl;
	  return -1;
	}
      }
    }

    std::cout << "chipID has been Loaded." << std::endl;
    return 0;
  }


  int Daq::DecodeSVXData(){
  
    static int s_decoded_index = 0;
    int re = 1;

    if(first_event){
      try{
	if(dummy_data)    read_bitdata(rawfile_name);
	else              re = read_svxdata();
      }catch(int e){
	std::cout << "Daq::DecodeSVXData() : "
		  << "Can't find first event data!!!" << std::endl;
	throw;
      }
      
      s_decoded_index = 0;
      // first_event = false;
    }
    
    init_vals();
    
    s_decoded_index = find_ev_header(s_decoded_index);
    s_decoded_index = decode_eventnum(s_decoded_index);
    s_decoded_index = decode_timestamp(s_decoded_index);
#ifdef NEW_HEADER_FORMAT
    s_decoded_index = decode_quality_info(s_decoded_index);
#endif
    
    while( !(found_chip[TELESCOPE_NUM - 1][3] || start_decode_next_ev) ){
      s_decoded_index = find_chip_header(s_decoded_index);
      s_decoded_index = decode_ch_adc(s_decoded_index);
    }
    
    return ev_num;
  }


  void Daq::init_vals(){

    found_tele_no = -1;
    found_chip_no = -1;
    start_decode_next_ev = false;

    m_ef_header.magicdata = 0;
    m_ef_header.totalsize = 0;
    m_ef_header.event_number = 0;
    m_ef_header.time_stamp = 0;
    m_ef_header.ef_number = 0;
#ifdef NEW_HEADER_FORMAT
    m_ef_header.data_quality = 0;
    m_ef_header.data_type = 0;
#endif
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){      
      // m_svxdata_buf[tele_i].event_number  = 111;
      // m_svxdata_buf[tele_i].time_stamp    = 222;
      // m_svxdata_buf[tele_i].status = 0;
      m_ef_header.ef_length[tele_i] = 0;

      m_svxdata_buf[tele_i].telescope_No  = 100;
      m_svxdata_buf[tele_i].hitnum        = 0;
      m_svxdata_buf[tele_i].svx_data.clear();
    
      for(int chip_i=0; chip_i<4; chip_i++){
	m_svxdata_buf[tele_i].chipID[chip_i] = 0;
	m_svxdata_buf[tele_i].pipelineID[chip_i] = 0;
	found_chip[tele_i][chip_i] = false;
      }
    }
  
  }


  int Daq::find_ev_header(int decoded_index){

    int data_cnt = decoded_index;
    int counter = 0;
    bool found_ev_header = false;
  
    while(counter < 1000){
    
      if(get_rawdata(data_cnt) == 0x00
	 && get_rawdata(data_cnt + 1) == 0x03
	 && get_rawdata(data_cnt + 2) == 0x06
	 && get_rawdata(data_cnt + 3) == 0x0d){
	found_ev_header = true;
	if(verbose_level > 0)
	  std::cout << std::endl
		    << data_cnt << " Event header is found." << std::endl;
	break;	
      }

      DataIncrement(data_cnt, 1);
      counter++;
    }

    if(counter > 5 && found_ev_header)
      write_errdata("Position of Event Header is something wrong.", data_cnt, counter, 15);

    if(counter == 1000)
      write_errdata("Event Header is not found...", data_cnt, counter, 15);
  
    DataIncrement(data_cnt, 4);
    return data_cnt;
  }


  int Daq::decode_eventnum(int decoded_index){
    static int prev_ev_num = -1;
    if(first_event){
      prev_ev_num = -1;
      first_event = false;
    }
    
    int data_cnt = decoded_index;
      
    ev_num = (256*256*256 * get_rawdata(data_cnt))
      + (256*256 * get_rawdata(data_cnt + 1))
      + (256 * get_rawdata(data_cnt + 2)) + get_rawdata(data_cnt + 3);
    m_ef_header.event_number = (unsigned int)ev_num;
    // for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++)
    //   m_svxdata_buf[tele_i].event_number = (unsigned int)ev_num;
  
    if(prev_ev_num != ev_num - 1){
      write_errdata("Event Number do NOT increment correctly.",
		    data_cnt, 10, 20);
    }
    
    if(verbose_level > 0)
      std::cout << data_cnt << " Event Number : " << ev_num << ", " << prev_ev_num << std::endl;

    prev_ev_num = ev_num;
  
    DataIncrement(data_cnt, 4);
      
    return data_cnt;
  }


  int Daq::decode_timestamp(int decoded_index){
  
    int data_cnt = decoded_index;
    Int_t timestamp = 0;
  
    timestamp = (256*256 * get_rawdata(data_cnt))
      + (256 * get_rawdata(data_cnt + 1))
      + get_rawdata(data_cnt + 2);
    m_ef_header.time_stamp = (unsigned int)timestamp;
    // for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++)
    //   m_svxdata_buf[tele_i].time_stamp = (unsigned int)timestamp;

    if(verbose_level > 0)
      std::cout << data_cnt << " Timestamp : " << timestamp << std::endl;
  
    DataIncrement(data_cnt, 3);
      
    return data_cnt;
  }

  
  int Daq::decode_quality_info(int decoded_index){

    int data_cnt = decoded_index;
    Int_t status = 0;

    status = get_rawdata(data_cnt);
#ifdef NEW_HEADER_FORMAT
    m_ef_header.data_quality = (unsigned char)status;
#endif
    // for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++)
    //   m_svxdata_buf[tele_i].status = (unsigned char)status;

    if(verbose_level > 0)
      std::cout << data_cnt << " Status : " << status << std::endl;

    DataIncrement(data_cnt,1);

    return data_cnt;
  }


  int Daq::find_chip_header(int decoded_index){

    int data_cnt = decoded_index;
    int counter = 0;
    bool fail_to_find = false;
  
    int next_tele_no = 0;
    int next_chip_no = 0;
    Short_t chipID = -1;
    Short_t pipeID = -1;
  
    // Next chip to be found
    for(int k=TELESCOPE_NUM*4; k>0; k--){
      if(found_chip[(k-1)/4][(k-1)%4] == true){
	next_tele_no = k/4;
	next_chip_no = k%4;
	break;
      }
    }

    while(counter < 5){
    
      if(get_rawdata(data_cnt) == chipID_config[next_tele_no][next_chip_no]){
	chipID = (Short_t)get_rawdata(data_cnt);
	pipeID = (Short_t)get_rawdata(data_cnt + 1);

	if(pipeID == 63){
	  std::cout << data_cnt << " Daq::find_chip_header() : "
		    << "WARNING:PipelineID is 63." << std::endl;
	}else if(pipeID > 46 || pipeID <= 0){
	  // fail_to_find = true; // To redecode
	  std::cout << data_cnt << " Daq::find_chip_header() : "
		    << "WARNING:PipelineID has illegal value("
		    << pipeID << ')' << std::endl;
	}
      
	break;
      }
	  
      DataIncrement(data_cnt, 1);
      counter++;
    }

    if(counter == 5 || fail_to_find){
      write_errdata("Failed to find Chip Header.",
		    data_cnt, 30, 30);
    
      if(data_cnt - decoded_index == counter)
	data_cnt = find_missing_chip_header(decoded_index);
      else
	data_cnt = find_missing_chip_header(-10);
    
    }else{
      if(verbose_level > 0)
	std::cout << data_cnt << " " << next_tele_no << " th telescope "
		  << next_chip_no << " th chip (chipID:" << chipID
		  << " pipelineID:" << pipeID << ')' << std::endl;
      found_tele_no = next_tele_no;
      found_chip_no = next_chip_no;
      found_chip[found_tele_no][found_chip_no] = true;
      m_svxdata_buf[found_tele_no].telescope_No = (unsigned short)(found_tele_no + 1);
      m_svxdata_buf[found_tele_no].chipID[found_chip_no] = (unsigned char)chipID;
      m_svxdata_buf[found_tele_no].pipelineID[found_chip_no] = (unsigned char)pipeID;
      DataIncrement(data_cnt, 2);
    }

    return data_cnt;
  }


  int Daq::find_missing_chip_header(int decoded_index){
    
    std::cout << decoded_index << " Daq::find_missing_chip_header() : Start."
	      << std::endl;
  
    int data_cnt = decoded_index;
    int tele_i, chip_i;
    Short_t chipID = -1;
    Short_t pipeID = -1;

    int counter = 0;
    bool do_break = false;
    bool fail_to_find = false;
    while(counter < 260){
    
      for(tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
	for(chip_i=0; chip_i<4; chip_i++){
	
	  if(get_rawdata(data_cnt) == chipID_config[tele_i][chip_i]
	     && found_chip[tele_i][chip_i] == false){
	  
	    chipID = (Short_t)get_rawdata(data_cnt);
	    pipeID = (Short_t)get_rawdata(data_cnt + 1);

	    if(pipeID != m_svxdata_buf[0].pipelineID[0]
	       && pipeID != m_svxdata_buf[tele_i].pipelineID[0]){
	      fail_to_find = true;
	      std::cout << data_cnt << " Daq::find_missing_chip_header() : "
			<< "WARNING:This Chip's pipelineID differ from the other chips."
			<< '(' << pipeID << ')' << std::endl;
	    }else{
	      if(pipeID == 63){
		std::cout << data_cnt << " Daq::find_missing_chip_header() : "
			  << "WARNING:PipelineID is 63." << std::endl;
	      }else if(pipeID > 46 || pipeID <= 0){
		std::cout << data_cnt << " Daq::find_missing_chip_header() : "
			  << "WARNING:PipelineID has illegal value("
			  << pipeID << ')' << std::endl;
	      }
	    }
	    do_break = true;
	  }
	  if(do_break) break;
	}
	if(do_break) break;
      }
      if(do_break) break;
    
      DataIncrement(data_cnt, 1);
      counter++;
    }

    if(counter == 260 || fail_to_find){
      std::cout << "Daq::find_missing_chip_header() : "
		<< "ERROR:Can't find missing chip header!" << std::endl;
    
      start_decode_next_ev = true;
      if(data_cnt - decoded_index == counter)
	data_cnt = decoded_index;
      else
	data_cnt = -10;
    
    }else{
      std::cout << data_cnt << " " << tele_i << " th telescope "
		<< chip_i << " th chip (chipID:" << chipID
		<< " pipelineID:" << pipeID << ')' << std::endl;
      found_tele_no = tele_i;
      found_chip_no = chip_i;
      found_chip[tele_i][chip_i] = true;
      m_svxdata_buf[found_tele_no].telescope_No = (unsigned short)(found_tele_no + 1);
      m_svxdata_buf[found_tele_no].chipID[found_chip_no] = (unsigned char)chipID;
      m_svxdata_buf[found_tele_no].pipelineID[found_chip_no] = (unsigned char)pipeID;
      DataIncrement(data_cnt, 2);
    }

    return data_cnt;
  }


  int Daq::decode_ch_adc(int decoded_index){

    if(start_decode_next_ev)
      return decoded_index;

    int data_cnt = decoded_index;
    int prev_ch_ID = -1;
    int ch_ID = -1;
    int adc_val = -1;
    int hit_num = 0;
    bool start_find_missing_ch = false;

    SVXdata tmp_svx_data;
  
    while(true){
      prev_ch_ID = ch_ID;
      ch_ID = get_rawdata(data_cnt);

      if(ch_ID >= 128){
	std::cout << data_cnt
		  << "Daq::decode_ch_adc() : "
		  << "WARNING:It is found that chID is over 128."
		  << '(' << ch_ID << ')' <<  std::endl;
	start_find_missing_ch = true;
	break;
      
      }else if(ch_ID <= prev_ch_ID){
	std::cout << data_cnt
		  << "Daq::decode_ch_adc() : "
		  << "WARNING:The order of chID is something wrong."
		  << '(' << ch_ID << ',' << prev_ch_ID << ')' << std::endl;
	start_find_missing_ch = true;
	break;
      
      }else if(hit_num > 128){
	std::cout << data_cnt
		  << "Daq::decode_ch_adc() : "
		  << "WARNING:More than 128chs are founds." << std::endl;
	start_find_missing_ch = true;
	break;
      }
    
      adc_val = (int)gray_decoder((unsigned char)get_rawdata(data_cnt+1));
      tmp_svx_data.hitch = (unsigned short)(ch_ID + 128*found_chip_no);
      tmp_svx_data.hitadc = (unsigned char)adc_val;
      m_svxdata_buf[found_tele_no].svx_data.push_back(tmp_svx_data);

      if(verbose_level > 1)
	std::cout << data_cnt << " ch:" << ch_ID << " adc:" << adc_val << std::endl;
      
      DataIncrement(data_cnt, 2);
      hit_num++;

      if(ch_ID == 127){
	if(verbose_level > 0){
	  std::cout << data_cnt - 2 << " ch:" << ch_ID
		    << " adc:" << adc_val << std::endl;
	  std::cout << "Daq::decode_ch_adc() : 127th ch is found." << std::endl;
	}
	m_svxdata_buf[found_tele_no].hitnum += (unsigned short)hit_num;
	break;
      }
    }

    if(start_find_missing_ch){
      write_errdata("Failed to decode Channnels and ADC Values.",
		    data_cnt, 30, 30);
      
      if(data_cnt - decoded_index == hit_num*2){
	for(int k=0; k<hit_num; k++)
	  m_svxdata_buf[found_tele_no].svx_data.pop_back();
	data_cnt = decode_missing_ch_adc(decoded_index);
      }else{
	for(int k=0; k<hit_num; k++)
	  m_svxdata_buf[found_tele_no].svx_data.pop_back();
	data_cnt = -10;
      }
      
    }
  
    return data_cnt;
  }


  int Daq::decode_missing_ch_adc(int decoded_index){
    std::cout << decoded_index << " Daq::find_missing_ch_adc() : Start." << std::endl;

    int data_cnt = decoded_index;
    int prev_ch_ID = -1;
    int ch_ID[2] = {-1, -1};
    int likelihood[2] = {-1, -1};
    int likely_ch_ID = -1;
    int adc_val = -1;
    int hit_num = 0;
    SVXdata tmp_svx_data;
  
    int counter = 0;
  
    while(true){
      prev_ch_ID = likely_ch_ID;
    
      for(int k=0; k<2; k++){
	ch_ID[k] = get_rawdata(data_cnt + k);
	likelihood[k] = likelihood_chID(ch_ID[k], prev_ch_ID, data_cnt + k);
      }

      if(likelihood[0] == -20 || likelihood[1] == -20){
	std::cout << data_cnt << " Daq::find_missing_ch_adc() : "
		  << "Parhap Found Event header." << std::endl;
	data_cnt = data_cnt - 2;
	break;
	
      }else if(likelihood[0] < 0 && likelihood[1] < 0){
	std::cout << data_cnt << " Daq::find_missing_ch_adc() : "
		  << "Channnel ADC Data has broken at all." << std::endl;
	m_svxdata_buf[found_tele_no].hitnum += (unsigned short)hit_num;
	data_cnt = data_cnt - 2;
	break;
	
      }else if(likelihood[0] >= 10 || likelihood[1] >= 10){
	
	if(likelihood[0] >= likelihood[1]){
	  likely_ch_ID = ch_ID[0];
	  adc_val = (int)gray_decoder((unsigned char)get_rawdata(data_cnt + 1));
	}else{
	  likely_ch_ID = ch_ID[1];
	  adc_val = (int)gray_decoder((unsigned char)get_rawdata(data_cnt + 2));
	  DataIncrement(data_cnt, 1);
	}

	tmp_svx_data.hitch = (unsigned short)(likely_ch_ID + 128*found_chip_no);
	tmp_svx_data.hitadc = (unsigned char)adc_val;
	m_svxdata_buf[found_tele_no].svx_data.push_back(tmp_svx_data);
	hit_num++;
	
	std::cout << data_cnt << " ch:" << likely_ch_ID << " adc:" << adc_val
		  << "  (" << ch_ID[0] << ',' << likelihood[0] << "), "
		  << '(' << ch_ID[1] << ',' << likelihood[1] << ')' << std::endl;
      }
    
      DataIncrement(data_cnt, 2);
      counter++;

      if(hit_num > 128){
	std::cout << data_cnt
		  << "Daq::decode_missing_ch_adc() : "
		  << "WARNING:More than 128chs are founds." << std::endl;
	
	for(int k=0; k<hit_num; k++) m_svxdata_buf[found_tele_no].svx_data.pop_back();
	start_decode_next_ev = true;
	
	if(data_cnt - decoded_index == hit_num*2)  data_cnt = decoded_index; // To Find Next Event
	else                                       data_cnt = -10;
	
	break;
	
      }else if(likely_ch_ID == 127){
	std::cout << data_cnt - 2 << " ch:" << likely_ch_ID
		  << " adc:" << adc_val << std::endl;
	std::cout << "Daq::decode_ch_adc() : 127th ch is found." << std::endl;
	m_svxdata_buf[found_tele_no].hitnum += (unsigned short)hit_num;
	break;
      }
    }
  
    return data_cnt;
  }


  int Daq::likelihood_chID(int chID, int prev_chID, const int data_cnt){
  
    int likelihood = 10;
    int next_chID = get_rawdata(data_cnt + 2);
    bool chipID_found = false;
    bool ev_header_found = false;

    if(get_rawdata(data_cnt) == 0x00
       && get_rawdata(data_cnt + 1) == 0x03
       && get_rawdata(data_cnt + 2) == 0x06
       && get_rawdata(data_cnt + 3) == 0x0d){
      ev_header_found = true;
      likelihood = -20;
      
    }else{
      
      for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++)
	for(int chip_i=0; chip_i<4; chip_i++)
	  if(chID == chipID_config[tele_i][chip_i]
	     && found_chip[tele_i][chip_i] == false)
	    chipID_found = true;

      if(chID == 127){
	if(prev_chID == 126)
	  likelihood = likelihood + 5;

	for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
	  for(int chip_i=0; chip_i<4; chip_i++){
	    if(next_chID == chipID_config[tele_i][chip_i]
	       && found_chip[tele_i][chip_i] == false)
	      likelihood = likelihood + 10;
	  }
	}
    
      }else if(chipID_found){
	int pipeID = get_rawdata(data_cnt + 1);
    
	if(next_chID == 0)
	  likelihood = likelihood - 5;

	if(get_rawdata(data_cnt - 2) == 127)
	  likelihood = likelihood - 7;

	if(pipeID == 63
	   || (pipeID > 0 && pipeID <= 46))
	  likelihood = likelihood - 7;
      
      }else if(chID >= 128)
	likelihood = likelihood - 5;
    
      if(chID <= prev_chID)
	likelihood = likelihood - 5;
    
      if(chID == prev_chID + 1)
	likelihood = likelihood + 3;
    
      if(chID == next_chID - 1)
	likelihood = likelihood + 3;
    }
  
    return likelihood;  
  }

} // namespace svx
