#include "svxcommon.h"
#include "util.h"

#include "SiTCP.h"
#include "comm.h"
#include "svxconfig.h"
#include "daq.h"
#include "analyze.h"
#include "logger.h"

#include <iostream>
#include <fstream>
#include <string>
#include <math.h>

#include "TTree.h"
#include "TFile.h"
#include "TStopwatch.h"

#include "stdio.h"
#include "stdlib.h"
#include "unistd.h"

#define IpAddr "192.168.10.16"
#define tcpport 24
#define udpport 4660

using namespace svx;

int DoTest(Test_Mode test_mode, Run_Mode run_mode);
int analyze_bitdata(SVXconfig* config, Analyze* analyzer,
		    Test_Mode test_mode, int iteration);
int take_data(SVXconfig* config, Analyze* analyzer,
	      Test_Mode test_mode, Run_Mode run_mode, int iteration);

int main(int argc, char *argv[]){
  getOptions(argc, argv);
  show_options();

  TStopwatch T;
  T.Start();
  
  if(g_run_mode != ANA)
    confirm_connection(IpAddr, tcpport, udpport);

  getchar(); // Debug

  DoTest(g_test_mode, g_run_mode);
  
  T.Stop();
  T.Print();

  return 0;
}


int DoTest(Test_Mode test_mode, Run_Mode run_mode){
  
  SVXconfig* config = new SVXconfig();
  Analyze* analyzer = new Analyze(test_mode);
  
  Load_all_configurations(config);

  // Change parameters written in option of ./run
  modify_conf_param(config, g_change_params);

  // std::cout << "Debug::rewrite_config_file()" << std::endl;
  // config->rewrite_config_file(test_mode);
  // getchar();
  
  // Change Configuration for test
  if(test_mode != Default){
    std::cout << std::endl << "Initialize configuration parameters for test..." << std::endl;
    analyzer->init_conf_for_test(config);
  }

  int test_result = -1;
  int iteration = 0;
  
  while(true){
    analyzer->show_message(iteration);
    
    analyzer->Reset_hists();
    
    if(run_mode == ANA)
      analyze_bitdata(config, analyzer, test_mode, iteration);
    else if(run_mode == DATA || run_mode == DATA_ANA){
      int ret = take_data(config, analyzer, test_mode, run_mode, iteration);
      if(ret == -1){
	std::cout << "WORNING : Error in SiTCP Connection or SVX4 configuration!"
		  << std::endl << std::endl;
	continue; // Connection ERROR
      }
    }

    test_result = analyzer->analyze_test_result(run_mode, iteration, g_change_params);
    
    std::cout << "Test Result : " << test_result << std::endl;
    
    // analyzer->show_test_result(test_mode, test_result);
    std::cout << std::endl << "Modify config parameters..." << std::endl;
    modify_conf_param(config, g_change_params);

    if(test_result == TEST_END)
      break;

    iteration++;
  }

  if(run_mode == DATA_ANA || run_mode == ANA){
    analyzer->write_test_result();
    if(test_mode != Default)
      analyzer->rewrite_config_file(config);
  }

  delete analyzer;
  delete config;

  return 0;
}


int analyze_bitdata(SVXconfig* config, Analyze* analyzer,
		    Test_Mode test_mode, int iteration){
  
  analyzer->SetFileName(get_rootfile_name(test_mode, iteration));
  Logger* logger = new Logger(get_treefile_name(test_mode, iteration));
  Daq* daq = new Daq(get_rawfile_name(test_mode, iteration));
  daq->get_config_param(config);
  analyzer->get_config_param_ana(config);

  std::cout << "+++++++++++++++++++ Analyze bitdata +++++++++++++++++++" << std::endl;

  int ev_cnt = 0;
  int decoded_evnum = 0;
  
  while(ev_cnt < g_num_event/*ev_cnt == decoded_evnum*/){

    try{
      decoded_evnum = daq->DecodeSVXData(); // Decode bitdata
    }catch(int e){
      std::cout << "main() : Retrying to find data...";
      continue;
    }catch(char* cstr){
      std::cout << cstr << std::endl; break;
    }
    
    char* data = (char*)malloc(daq->GetDecodedDataSize());
    daq->GetDecodedData(data);
    
    logger->FillTBranches(data);
    analyzer->analyze_data(data);
    
    if(ev_cnt%1000 == 999)
      analyzer->write_rootfile();
    
    free(data);
    ev_cnt++;
  }
  std::cout << "++++++++++++++++++ Analysis Finished ++++++++++++++++++" << std::endl;
  
  delete logger;
  delete daq;
  
  return 0;
}


int take_data(SVXconfig* config, Analyze* analyzer,
	      Test_Mode test_mode, Run_Mode run_mode, int iteration){

  analyzer->SetFileName(get_rootfile_name(test_mode, iteration));
  Logger* logger = new Logger(get_treefile_name(test_mode, iteration));
  Comm* comm = new Comm();
  Daq* daq = new Daq(comm, get_rawfile_name(test_mode, iteration));
  
  comm->SetIPPort(IpAddr, tcpport, udpport);
  comm->CreateUDPSock();
  // comm->CreateTCPSock();
  comm->clear_all(); // clear all registers in FPGA
  
  // Activate FPGA
  comm->udp_send(COMMAND, COM_ACTIVE);
  std::cout << "FPGA has been activated." << std::endl;
  
  std::cout << "Sending configuration data to FPGA..." << std::endl;
  SetParstVetoParam(comm, config->getparam(PickDel, 0, 0, 0), 35, 40);
  SetUseTele(comm, TELESCOPE_NUM);
  
  for(int j=0; j<TELESCOPE_NUM; j++)
    config->send_config(comm, j);
    
  config->send_ctrlsig(comm);
  std::cout << "Sended configuration data to FPGA!!!" << std::endl;

  for(int param=0; param<PARAM_INDEX_END; param++){
    std::cout << Strparam_names((param_names)param) << std::endl;
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      for(int chip_i=0; chip_i<4; chip_i++)
	config->getparam((param_names)param, tele_i, chip_i, 1);
      std::cout << std::endl;
    }
    // std::cout << std::endl;
  }

  std::cout<<std::endl
	   <<"***** Communication time *****" <<std::endl
	   <<"         SEABAS vs PC         " <<std::endl
	   <<"******************************" <<std::endl
	   <<std::endl;
  
  comm->CreateTCPSock();

  std::cout << "Sending configuration data to SVX4..." << std::endl;
  
  while(true){
    std::cout << "Press Enter key." << std::endl;
    // getchar();
    try{
      int conf_re = config->send_config_to_svx(comm);
      if(conf_re == 0)  break;
    }
    catch(int e){
      comm->clear_all();
      comm->CloseTCPSock();
      comm->CloseUDPSock();
  
      delete logger;
      delete daq;
      delete comm;
      
      return -1; // Connection ERROR // To be improve
    }
  }
  
  usleep(1000);
  std::cout << "Sended configuration data to SVX4!!!" << std::endl;

  daq->get_config_param(config);
  analyzer->get_config_param_ana(config);

  std::cout << "Press Enter key." << std::endl;
  getchar();
  
  std::cout << "++++++++++++++++++++ DAQ Start ++++++++++++++++++++" << std::endl;
  comm->udp_send(COMMAND, COM_START); //start DAQ
    
  int ev_cnt = 0;
  int decoded_evnum = 0;
  
  TStopwatch DAQtime;
  DAQtime.Start();
  PrintCurrentTime();
  
  while(ev_cnt < g_num_event){

    try{
      decoded_evnum = daq->DecodeSVXData(); // Decode rawdata
    }catch(int e){
      std::cout << "main() : Retrying to find data...";
      continue;
    }catch(char* cstr){
      std::cout << cstr << std::endl; break;
    }
    
    char* data = (char*)malloc(daq->GetDecodedDataSize());
    daq->GetDecodedData(data);
    
    logger->FillTBranches(data);

    if(run_mode != DATA){
      analyzer->analyze_data(data);

      // if(ev_cnt%10 == 9)
      // logger->write_rootfile();
      if(ev_cnt%100 == 99){
    	analyzer->write_rootfile();
	std::cout << "Rest of Events == " << g_num_event - ev_cnt << std::endl;;
      }
    }
    
    free(data);
    ev_cnt++;
  }
  DAQtime.Stop();
  DAQtime.Print("u");
  std::cout << "++++++++++++++++++++ DAQ Finished ++++++++++++++++++++" << std::endl;

  analyzer->write_rootfile();

  // Finalize
  comm->udp_send(COMMAND, COM_STOP);
  comm->udp_send(COMMAND, COM_UNCONFIGURE);
  comm->udp_send(COMMAND, COM_DEACTIVE);
  comm->clear_all();
  comm->CloseTCPSock();
  comm->CloseUDPSock();
  
  delete logger;
  delete daq;
  delete comm;

  return 0;
}
