#include "util.h"

#include <iostream>
#include <fstream>
#include <string>
#include <sstream>

#include "stdio.h"
#include "stdlib.h"
#include "unistd.h"
#include "time.h"

namespace svx{

  void PrintCurrentTime(){
    time_t current_time;
    time(&current_time);
    std::cout << std::endl
	      << ctime(&current_time)
	      << std::endl;
  }
  
  int getOptions(int argc, char *argv[]){
    bool outputfile_flag = false;
    
    std::cout << std::endl;
  
    if(argc == 1){
      std::cout << "No option. Start as Default Mode." << std::endl
		<< std::endl;
      return 0;
    }else{
    
      for(int arg_i=1; arg_i<argc; arg_i++){

	// help
	if(strncmp(argv[arg_i], "--help\0", 7) == 0
	   || strncmp(argv[arg_i], "-h\0", 3) == 0)
	  show_help();
      
	// run mode
	else if(strncmp(argv[arg_i], "--mode\0", 7) == 0
		|| strncmp(argv[arg_i], "-m\0", 3) == 0){
	  if(arg_i+1 < argc
	     && strncmp(argv[arg_i+1], "-", 1) != 0
	     && (atoi(argv[arg_i+1]) > 0 && atoi(argv[arg_i+1]) < 4)){
	    g_run_mode = (enum Run_Mode)atoi(argv[arg_i+1]);
	    arg_i++;
	  }else{
	    std::cout << "Can't find valid argument of Run Mode."
		      << std::endl << std::endl;
	    show_help();
	  }
	}
      
	// Event Number
	else if(strncmp(argv[arg_i], "--event\0", 8) == 0
		|| strncmp(argv[arg_i], "-e\0", 3) == 0){
	  if(arg_i+1 < argc
	     && strncmp(argv[arg_i+1], "-", 1) != 0
	     && atoi(argv[arg_i+1]) > 0){
	    g_num_event = atoi(argv[arg_i+1]);
	    arg_i++;
	  }else{
	    std::cout << "Can't find valid argument \"<EVENT_NUMBER>\" ."
		      << std::endl << std::endl;
	    show_help();
	  }
	}
      
	// input file
	else if(strncmp(argv[arg_i], "--input\0", 8) == 0
		|| strncmp(argv[arg_i], "-i\0", 3) == 0){
	  if(arg_i+1 < argc && strncmp(argv[arg_i+1], "-", 1) != 0){
	    // listName = argv[arg_i+1];
	    arg_i++;
	  } else{
	    std::cout << "Can't find input file." << std::endl << std::endl;
	    show_help();
	  }
	}

	// output file
	else if(strncmp(argv[arg_i], "--output\0", 8) == 0
		|| strncmp(argv[arg_i], "-o\0", 3) == 0){
	  if(arg_i+1 < argc && strncmp(argv[arg_i+1], "-", 1) != 0){
	    g_filename = argv[arg_i+1];
	    arg_i++;
	    outputfile_flag = true;
	  } else{
	    std::cout << "Can't find input file." << std::endl << std::endl;
	    show_help();
	  }
	}
	
	// verbose level
	else if(strncmp(argv[arg_i], "--verbose\0", 10) == 0
		|| strncmp(argv[arg_i], "-v\0", 3) == 0){
	  if(arg_i+1 < argc && strncmp(argv[arg_i+1], "-", 1) != 0
	     && (atoi(argv[arg_i+1]) > 0 || strncmp(argv[arg_i+1], "0\0", 2) == 0)
	     && atoi(argv[arg_i+1]) < 4){
	    g_verbose_level = atoi(argv[arg_i+1]);
	    arg_i++; 
	  }else{
	    std::cout << "illegal verbose level." << std::endl << std::endl;
	    show_help();
	  }
	}

	// batch
	// }else if(strncmp(argv[arg_i], "--batch", 7) == 0 || strncmp(argv[arg_i], "-b", 2) == 0){
	
	// }
      
	// <TEST_IDENTIFIER>	
	else if(arg_i == 1
		&& (atoi(argv[arg_i]) > 0 && atoi(argv[arg_i]) < 6)){
	  g_test_mode = (enum Test_Mode)atoi(argv[arg_i]);
	}

	// Configuration Parameter
	else{
	  bool found_param = false;
	  for(int i=0; i<PARAM_INDEX_END; i++){
	    std::string tmp_str = "--";
	    tmp_str += Strparam_names((param_names)i);
	    tmp_str.erase(tmp_str.find_first_of(' '));
	    if(strcmp(tmp_str.c_str(), argv[arg_i]) == 0){
	      if(arg_i+1 < argc
		 && strncmp(argv[arg_i+1], "-", 1) != 0
		 && atoi(argv[arg_i+1]) >= 0){
		change_param tmp_param
		  = {(param_names)i, atoi(argv[arg_i+1]), 0, 0, true};
		g_change_params.push_back(tmp_param);
		  
		found_param = true;
		arg_i++;
	      }else{
		std::cout << "Can't find valid configuration parameter."
			  << std::endl << std::endl;
		show_help();
	      }
	    }
	  }
	
	  if(!found_param){
	    std::cout << "illegal option -- " << argv[arg_i] << std::endl
		      << std::endl;
	    show_help();
	  }
	}
      } // for loop end
    }

    if(!outputfile_flag){
      std::cout << "OUTPUT FILENAME!!!" << std::endl;
      show_help();
    }
      
  
    return 0;
  }


  void show_options(){
  
    std::cout << "  - Run mode        = ";
    switch(g_run_mode){
    case 1:  std::cout << "DATA           " << std::endl; break;
    case 2:  std::cout << "ANALYSIS       " << std::endl; break;
    case 3:  std::cout << "DATA + ANALYSIS" << std::endl; break;
    default: std::cout << "** Unknown run_mode " << g_run_mode << " !!!" << std::endl; break;
    };
  
    std::cout << "  - Test mode       = ";
    switch(g_test_mode){
    case 0:  std::cout << "Default DAQ       " << std::endl; break;
    case 1:  std::cout << "Mask Scan         " << std::endl; break;
    case 2:  std::cout << "Gain Measurement  " << std::endl; break;
    case 3:  std::cout << "Threshold Scan    " << std::endl; break;
    case 4:  std::cout << "PickDel Scan      " << std::endl; break;
    case 5:  std::cout << "Trigger Delay Scan" << std::endl; break;
    default: std::cout << "** Unknown run_mode " << g_run_mode << " !!!" << std::endl; break;
    };

    std::cout << "  - Number of event = " << g_num_event << std::endl;
    // std::cout << "  - Input file      = " <<  << std::endl;
    // std::cout << "  - Output files    = " <<  << std::endl;
  
  }


  void show_help(){
    std::cout << "Usage: ./run <TEST_IDENTIFIER> [OPTION(s)] " << std::endl
	      << std::endl;
    std::cout << " where TEST_IDENTIFIER corresponds to one of the tests listed below:      " << std::endl
	      << "       1 : Mask scan                          " << std::endl
	      << "       2 : Gain measurement                   " << std::endl
	      << "       3 : Threshold scan                     " << std::endl
	      << "       4 : Pickdel scan                       " << std::endl
	      << "       5 : Trigger delay scan                 " << std::endl
	      << std::endl;
    std::cout << "Optional arguments:" << std::endl
	      << std::endl;
    std::cout << "   -m, --mode <MODE>               Run mode: 1:DATA, 2:ANALYSIS, 3:DATA+ANALYSIS (default = 3)." << std::endl
	      << std::endl;
    // std::cout << "   -b, --batch                     Run in batch mode, i.e, no plots shown (default = false)." << std::endl
    //   	    << std::endl;
    std::cout << "   -i, --input <DIRECTORY>         Reads configuration files (config_tele*.dat)" << std::endl
	      << "                                   in <DIRECTORY> (default = ../text/)" << std::endl
	      << std::endl;
    std::cout << "   -o, --output <NAME>             Name of output files." << std::endl
	      << std::endl;
    std::cout << "   -e, --event <EVENT_NUMBER>      Sets the number of events taken (default = 1000)." << std::endl
	      << std::endl;
    std::cout << "   -v, --verbose <VERBOSE LEVEL>   Sets the verbosity level from 0 to 3 included (default = 1)." << std::endl
	      << std::endl;
    std::cout << "   --<PARAM_NAME> <VALUE>          Sets the config parameter to <PARAM_NAME> " << std::endl
	      << "                                   of the all telescopes <VALUE> forcely." << std::endl
	      << std::endl;
    std::cout << "   -h, --help                      Displays this information and exit." << std::endl
	      << std::endl;
  
    std::cout << "Example: ./run 2 -m 2 -Thresh 40 -v 2" << std::endl
	      << "  will set config paramerter Thresh to 40, set verbosity level to 2" << std::endl
	      << "  and analysize the data from gain mesurement test" << std::endl
	      << std::endl;
    exit(0);
  } 


  void modify_conf_param(SVXconfig* config, std::vector<struct change_param> &change_params){
    if(change_params.empty())
      std::cout << "modify_conf_param() : "
		<< "No change configuration parameters." << std::endl;
    else{
      for(int i=0; i<(int)change_params.size(); i++){
	if(change_params[i].change_all)
	  config->writeparam_allchips(change_params[i].name, change_params[i].val);
	else
	  config->writeparam(change_params[i].name, change_params[i].tele_i,
			     change_params[i].chip_i, change_params[i].val, 1);
      }
    
      change_params.clear();
    }
  }


  void confirm_connection(const char* IpAddr, unsigned int tcp, unsigned int udp){
  
    std::cout << std::endl
	      << "========== Confirmation connection between SEABAS and PC ==========" << std::endl;
    
    Comm* comm_test = new Comm();
    comm_test->SetIPPort((char*)IpAddr, tcp, udp);
    comm_test->CreateUDPSock();
    comm_test->CreateTCPSock();
    
    show_firmware_info(comm_test);
    // int msg_len = 0;
    // comm_test->read_tcp_data(msg_len, NULL, 1000);

    comm_test->CloseTCPSock();
    comm_test->CloseUDPSock();
    
    delete comm_test;
    std::cout << std::endl
	      << "================ Finish Confirmation the connection ===============" << std::endl
	      << std::endl;
    
  }


  void show_firmware_info(Comm* comm){
    unsigned char udpdata[255];
    comm->read_udp_data(0x100, udpdata, 255, 4, 0);

    std::cout << "+=+=+=+=+=+=+=+=+=+=+ Firmware Version Info +=+=+=+=+=+=+=+=+=+=+" << std::endl
	      << "Date: " << std::hex << (int)udpdata[0] << "/" << (int)udpdata[1]
	      << "/20" <<  (int)udpdata[2] << std::endl
	      << "Version: " << std::dec << (int)udpdata[3] << std::endl
	      << std::endl;
  }


  void Load_all_configurations(SVXconfig* config){
  
    std::cout << std::endl
	      << "========== Start loading configuration files... ==========" << std::endl
	      << std::endl;
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++)
      config->load_config(tele_i);

    config->load_ctrlsig();
  
    std::cout << std::endl
	      <<"========== Finish loading configuration files... =========" << std::endl
	      << std::endl;
  }


  void SetParstVetoParam(Comm* comm, unsigned char pickdel, unsigned char latency,
			 unsigned char width, bool scanmode){
    unsigned char scan = (scanmode) ? 0x1<<6 : 0;
    comm->udp_send(0x105, pickdel + scan);
    comm->udp_send(0x106, latency); // latency
    comm->udp_send(0x107, width); // Mask length
  }


  void SetUseTele(Comm* comm, int tele_num){
    unsigned char data = 0x00;
    for(int tele=0; tele<tele_num; tele++)
      data += 0x01<<tele;

    comm->udp_send(0x108, data);
  }


  std::string get_rawfile_name(Test_Mode test_mode, int iteration){
    std::ostringstream rawfile_name;
#ifdef SCTJDAQ_COMPILE
    rawfile_name << "/home/sctjdaq/data";
#endif
    rawfile_name << "../data/";
  
    switch(test_mode){
    case Mask:
      rawfile_name << "Mask/mask_rawdata_" << iteration;
      break;
    case Gain:
      rawfile_name << "Gain/gain_rawdata_" << iteration;
      break;
    case Threshold:
      rawfile_name << "Threshold/thresh_rawdata_" << iteration;
      break;
    case Pickdel_scan:
      rawfile_name << "Pickdel_scan/pickdel_rawdata_" << iteration;
      break;
    case Trig_delay:
      rawfile_name << "Trig_deley/delay_rawdata_" << iteration;
      break;
    default:
      rawfile_name << g_filename << "_raw";
      break;
    }
  
    rawfile_name << ".dat";
    return rawfile_name.str();
  }

 
  std::string get_rootfile_name(Test_Mode test_mode, int iteration){
    std::ostringstream rootfile_name;
#ifdef SCTJDAQ_COMPILE
    rootfile_name << "/home/sctjdaq/data/";
#endif
    rootfile_name << "../data/";
  
    switch(test_mode){
    case Mask:
      rootfile_name << "Mask/mask_ana_" << iteration;
      break;
    case Gain:
      rootfile_name << "Gain/gain_ana_" << iteration;
      break;
    case Threshold:
      rootfile_name << "Threshold/thresh_ana_" << iteration;
      break;
    case Pickdel_scan:
      rootfile_name << "Pickdel_scan/pickdel_ana_" << iteration;
      break;
    case Trig_delay:
      rootfile_name << "Trig_deley/delay_ana_" << iteration;
      break;
    default:
      rootfile_name << g_filename << "_ana";
      break;
    }
  
    rootfile_name << ".root";
    return rootfile_name.str();
  }


  std::string get_treefile_name(Test_Mode test_mode, int iteration){
    
    std::ostringstream treefile_name;
#ifdef SCTJDAQ_COMPILE
    treefile_name << "/home/sctjdaq/data/";
#endif
    treefile_name << "../data/";
  
    switch(test_mode){
    case Mask:
      treefile_name << "Mask/mask_tree_" << iteration;
      break;
    case Gain:
      treefile_name << "Gain/gain_tree_" << iteration;
      break;
    case Threshold:
      treefile_name << "Threshold/thresh_tree_" << iteration;
      break;
    case Pickdel_scan:
      treefile_name << "Pickdel_scan/pickdel_tree_" << iteration;
      break;
    case Trig_delay:
      treefile_name << "Trig_deley/delay_tree_" << iteration;
      break;
    default:
      treefile_name << g_filename << "_tree";
      break;
    }
  
    treefile_name << ".root";
    return treefile_name.str();
  }
  
} // namespace svx
