#include "analyze.h"

#include "stdio.h"
#include "stdlib.h"
#include "unistd.h"

#include <math.h>

#define SVX4_MAGIC_NUM 25104 //magic number

namespace svx{

  Analyze::Analyze(Test_Mode test_mode){

    m_test_mode = test_mode;
    num_event = g_num_event;
    
#ifdef SCTJDAQ_COMPILE
    filename = "/home/sctjdaq/data/svx_ana.root";
#else
    filename = "/home/sctjdaq/data/svx_ana.root"; // Default Filename
#endif
    rest_data_exist = false;
    Mask_mode = true;

    ts_log = 0;

    Int_t ADC_MAX = 260;
    Int_t ADC_MIN = 0;

    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      for(int chip_i=0; chip_i<4; chip_i++){
	
	for(int i=0; i<16; i++)
	  mask_config[tele_i][chip_i][i] = 0;

	for(int ch=0; ch<128*TELESCOPE_NUM; ch++){
	  masked_ch[tele_i][ch] = false;
	  prev_masked_ch[tele_i][ch] = false;
	}
      
	chipID_config[tele_i][chip_i] = -1;
	thresh_config[tele_i][chip_i] = -1;
      }
    }
  
    hist_time_stamp = new TH1I("TS_Diff_SVX4",
			       "timestamp difference of SVX4;Difference of Timestamp;Count/bin",
			       20500, -0.5, 20499.5);
  
    SVX4_TSvsEv = new TH1I("TS_vs_EV_SVX4", "Time stamp vs Event;Event;Timestamp",
			   100000., -0.5, 99999.5);
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      hist_svx_Front[tele_i] = new TH2I(Form("ADC_vs_CH_F_%d",tele_i),
					Form("hist_svx_Front_chip_tele%d;ch;ADC",tele_i),
					256, 0, 256, ADC_MAX-ADC_MIN+1, ADC_MIN, ADC_MAX);
      hist_svx_Rear[tele_i] = new TH2I(Form("ADC_vs_CH_R_%d",tele_i),
				       Form("hist_svx_Rear_chip_tele%d;ch;ADC",tele_i), 
				       256, 0, 256, ADC_MAX-ADC_MIN+1, ADC_MIN, ADC_MAX);
    
      hitmap_svx[tele_i] = new TH2I(Form("Hitmap_%d",tele_i),
				    Form("Hitmap_svx_tele%d;ch;ch",tele_i),
				    256,0,256,256,0,256);
    
      hist_pipelineID_svx_Front[tele_i] = new TH2I(Form("pipeID_F_%d",tele_i),
						   Form("pipeID_svx_Front_chip_tele%d;ch;PipeID",tele_i),
						   256, 0, 256, 47, 0, 47);
    
      hist_pipelineID_svx_Rear[tele_i] = new TH2I(Form("pipeID_R_%d",tele_i),
						  Form("pipeID_svx_Rear_chip_tele%d;ch;PipeID",tele_i),
						  256, 0, 256, 47, 0, 47);
    
      event_valid[tele_i] = new TH1I(Form("EvValid_%d", tele_i), Form("Event valid tele%d", tele_i),
				     10000, -0.5, 9999.5);
  
      for(int tele_j=tele_i; tele_j<TELESCOPE_NUM; tele_j++){
	correlation_svx_Front[tele_i][tele_j] = new TH2I(Form("corr_F_%d_%d",tele_i,tele_j),
							 Form("correlation_svx_Front_chips_tele_%d_vs_tele%d",tele_i,tele_j),
							 256, 0, 256, 256, 0, 256);
	correlation_svx_Front[tele_i][tele_j]->SetXTitle(Form("tele%d_ch", tele_i));
	correlation_svx_Front[tele_i][tele_j]->SetYTitle(Form("tele%d_ch", tele_j));

	correlation_svx_Rear[tele_i][tele_j] = new TH2I(Form("corr_R_%d_%d",tele_i,tele_j),
							Form("correlation_svx_Rear_chips_tele_%d_vs_tele%d",tele_i,tele_j),
							256, 0, 256, 256, 0, 256);
	correlation_svx_Rear[tele_i][tele_j]->SetXTitle(Form("tele%d_ch", tele_i));
	correlation_svx_Rear[tele_i][tele_j]->SetYTitle(Form("tele%d_ch", tele_j));
      }
    }

    create_test_object(test_mode);

    // Member variables for test_mode
    dead_ch_num = 0;
  }
  

  Analyze::~Analyze(){
    if(rest_data_exist)
      write_rootfile();
  
    delete hist_time_stamp;
    delete SVX4_TSvsEv;
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      delete hist_svx_Front[tele_i];
      delete hist_svx_Rear[tele_i];
      delete hitmap_svx[tele_i];
      delete hist_pipelineID_svx_Front[tele_i];
      delete hist_pipelineID_svx_Rear[tele_i];
      delete event_valid[tele_i];
      for(int tele_j=tele_i; tele_j<TELESCOPE_NUM; tele_j++){
	delete correlation_svx_Front[tele_i][tele_j];
	delete correlation_svx_Rear[tele_i][tele_j];
      }
    
      // delete_test_object(test_mode);
      delete mask_pattern[tele_i];
      delete pedestal_mean_Front[tele_i];
      delete pedestal_mean_Rear[tele_i];
      for(int num=0; num<32; num++){
	delete gain_curve_Front[tele_i][num];
	delete gain_curve_Rear[tele_i][num];
      }
    }
    
    delete hitnum_vs_pickdel;
    delete hitnum_vs_delay;
  }


  int Analyze::unpack_data(const char *data){
  
    const char *buf_pointer;

    memcpy((char*)&ef_header_ana, data, sizeof(SVX_EF_HEADER));
    buf_pointer = data + sizeof(SVX_EF_HEADER);

    // if(g_verbose_level > 0){
    std::cout << "Magic Data : " << (int)ef_header_ana.magicdata << std::endl
	      << "Total Length of this data : " << (int)ef_header_ana.totalsize << std::endl
	      << "Number of Telescope : " << (int)ef_header_ana.ef_number << std::endl
	      << "Length from data of Tele1 : " << (int)ef_header_ana.ef_length[0] << std::endl
	      << "Length from data of Tele2 : " << (int)ef_header_ana.ef_length[1] << std::endl
	      << "Length from data of Tele3 : " << (int)ef_header_ana.ef_length[2] << std::endl
	      << "Length from data of Tele4 : " << (int)ef_header_ana.ef_length[3] << std::endl
	      << "Event_number = " << (int)ef_header_ana.event_number << std::endl
	      << "Time_stamp = " << (int)ef_header_ana.time_stamp << std::endl;
#ifdef NEW_HEADER_FORMAT
    std::cout << "Data_quality = " << (int)ef_header_ana.data_quality << std::endl
	      << "Data_type = " << (int)ef_header_ana.data_type << std::endl;
#endif
      // }
  
    for(int tele_i=0; tele_i<ef_header_ana.ef_number; tele_i++){
      int data_size = (int)ef_header_ana.ef_length[tele_i];
      svx_ef_ana[tele_i] = (SVX_EF*)malloc(data_size);
    
      std::cout << "telescope No. " << tele_i << std::endl
		<< "data_size = " << data_size << std::endl;
    
      if(svx_ef_ana == NULL){
	std::cout << "Memory Allocaiton Error! Telescope data" << tele_i << std::endl;
	return -1;
      }
 
      memcpy((char*)svx_ef_ana[tele_i], buf_pointer , data_size);
      buf_pointer += data_size; // Shift Pointer to Next EF
    }
  
    return 0;
  }


  void Analyze::create_test_object(Test_Mode test_mode){
    
    if(test_mode == Mask){
      for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
	mask_pattern[tele_i] = new TH2C(Form("mask_pattern_%d",tele_i),
					Form("Mask_pattern_tele%d;ch;ch",tele_i),
					256,0,256,256,0,256);
      }
    
    }else if(test_mode == Gain){
      for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
	for(int num=0; num<32; num++){
	  gain_curve_Front[tele_i][num] = new TGraphErrors(5);
	  gain_curve_Front[tele_i][num]
	    ->SetNameTitle(Form("Gcurve_F_tele%d_ch%d",tele_i,num*8+4),
			   Form("Gain_Curve_tele%d_F_ch%d;Volt;ADCmean",tele_i,num*8+4));
	  gain_curve_Rear[tele_i][num] = new TGraphErrors(5);
	  gain_curve_Rear[tele_i][num]
	    ->SetNameTitle(Form("Gcurve_R_tele%d_ch%d",tele_i,num*8+4),
			   Form("Gain_Curve_tele%d_R_ch%d;Volt;ADCmean",tele_i,num*8+4));
	}
      }
      
    }else if(test_mode == Threshold){
      for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
	pedestal_mean_Front[tele_i] = new TH1D(Form("pedestal_F_%d", tele_i),
					       Form("Pedestal_Mean_F_tele%d;ch;ADC", tele_i),
					       256, 0, 256);
	pedestal_mean_Rear[tele_i] = new TH1D(Form("pedestal_R_%d", tele_i),
					      Form("Pedestal_Mean_R_tele%d;ch;ADC", tele_i),
					      256, 0, 256);
      }
      
    }else if(test_mode == Pickdel_scan){
      hitnum_vs_pickdel = new TGraphErrors(30);
      hitnum_vs_pickdel->SetNameTitle("hitnum_vs_pickdel", "Hit_Number_vs_PickDel");

    }else if(test_mode == Trig_delay){
      hitnum_vs_delay = new TGraphErrors(10);
      hitnum_vs_delay->SetNameTitle("hitnum_vs_delay", "Hit_Number_vs_Delay");
    }

    return;
  }


  int Analyze::fill_adc_vs_ch(const int tele_i,
			      int &most_hitted_ch_cnt_F, int &most_hitted_ch_cnt_R){

    most_hitted_ch_cnt_F = -1;
    most_hitted_ch_cnt_R = -1;
    int most_hitted_ch_adc_F = 0;
    int most_hitted_ch_adc_R = 0;
    
    for(int n=0; n<svx_ef_ana[tele_i]->hitnum; n++){
    
      int hit_ch = (int)svx_ef_ana[tele_i]->svx_data[n].hitch;
      int hit_adc = (int)svx_ef_ana[tele_i]->svx_data[n].hitadc;
      int chip_i = hit_ch/128;

      if(not(hit_ch >=0 && hit_ch < 512)){
	std::cout << "illegal hit channel(" << hit_ch << ')' << std::endl;
	
      }else{
	if(hit_adc > 255 || hit_adc < 0){
	  std::cout << "ch:" << hit_ch << " hitadc has invalid value.(" << hit_adc << ')' << std::endl;
	}else{
	  if(disable_config[tele_i][chip_i] == 1
	     && masked_ch[tele_i][hit_ch] && Mask_mode){
	    // std::cout << "ch:" << hit_ch << " masked channel." << std::endl;
	  }else{
	    
	    // if(hit_adc == 255)
	    //   std::cout << "ch:" << hit_ch << " hitadc is 255." << std::endl; // Debug

	    if(hit_ch < 256){
	      hist_svx_Front[tele_i]->Fill(hit_ch, hit_adc);
	      hist_pipelineID_svx_Front[tele_i]
		->Fill(hit_ch, svx_ef_ana[tele_i]->pipelineID[chip_i]);
	
	      if(hit_adc >= thresh_config[tele_i][chip_i]){
		// Hit! (Front SVX chips)
		// std::cout << "hit! ch:" << hit_ch << " adc:" << hit_adc << std::endl;
		if(hit_adc >= most_hitted_ch_adc_F && hit_adc<255){
		  most_hitted_ch_cnt_F = n;
		  most_hitted_ch_adc_F = hit_adc;
		}
	      }
	      
	    }else if(hit_ch >= 256 && hit_ch < 512){
	      hist_svx_Rear[tele_i]->Fill(hit_ch-256, hit_adc);	
	      hist_pipelineID_svx_Rear[tele_i]
		->Fill(hit_ch-256, svx_ef_ana[tele_i]->pipelineID[chip_i]);
	  
	      if(hit_adc >= thresh_config[tele_i][chip_i]){
		// Hit! (Rear SVX chips)
		// std::cout << "hit! ch:" << hit_ch << " adc:" << hit_adc << std::endl;
		if(hit_adc >= most_hitted_ch_adc_R && hit_adc<255){
		  most_hitted_ch_cnt_R = n;
		  most_hitted_ch_adc_R = hit_adc;
		}
	      }
	    }
	    
	  }
	}
      }
    }
  
    return 0;
  }
  

  int Analyze::fill_timestamp(){
    int time_stamp_diff;
    // static int ts_log = 0;
  
    time_stamp_diff = ef_header_ana.time_stamp - ts_log;
    ts_log = ef_header_ana.time_stamp;
  
    std::cout << "timestamp = " << ts_log << std::endl;
    std::cout << "differencial of timestamp = " << time_stamp_diff << std::endl;
  
    hist_time_stamp->Fill(time_stamp_diff);
    SVX4_TSvsEv->Fill(ef_header_ana.event_number,ef_header_ana.time_stamp);

    return 0;
  }


  int Analyze::fill_ev_valid(const int tele_i){
    // pipelineID check
    int event_validity = 10;
    for(int chip_i=0; chip_i<4; chip_i++){
      if(svx_ef_ana[tele_i]->pipelineID[chip_i] > 46 || svx_ef_ana[tele_i]->pipelineID[chip_i] < 1)
	event_validity = event_validity - 2;
    }
    
    //hitnum check
    if(svx_ef_ana[tele_i]->hitnum > 512)
      event_validity = event_validity - 1;
    
    event_valid[tele_i]->Fill(ef_header_ana.event_number,event_validity);
    
    if(event_validity != 10)
      return event_validity;

    return 0;
  }


  int Analyze::analyze_data(const char *data){

    unpack_data(data);

    std::cout<<"Fill start"<<std::endl;
    rest_data_exist = true;

    int tele_num = ef_header_ana.ef_number;
    int ev_num = ef_header_ana.event_number;
  
    fill_timestamp();
  
    for(int tele_i=0; tele_i<tele_num; tele_i++)
      fill_ev_valid(tele_i);

    ////////////////////////////////////
    // make hitmap and adc histgrams
    int most_hitch_cnt_F[TELESCOPE_NUM] = {0};
    int most_hitch_cnt_R[TELESCOPE_NUM] = {0};
    int adc_max_ch_F[TELESCOPE_NUM] = {0};
    int adc_max_ch_R[TELESCOPE_NUM] = {0};
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
    
      std::cout << tele_i << "th tele's the number of hits = "
		<< (int)(svx_ef_ana[tele_i]->hitnum) << std::endl;

      fill_adc_vs_ch(tele_i, most_hitch_cnt_F[tele_i], most_hitch_cnt_R[tele_i]);
    
      if(most_hitch_cnt_F[tele_i] != -1)
	adc_max_ch_F[tele_i]
	  = (int)svx_ef_ana[tele_i]->svx_data[most_hitch_cnt_F[tele_i]].hitch;
      
      if(most_hitch_cnt_R[tele_i] != -1)
	adc_max_ch_R[tele_i]
	  = (int)svx_ef_ana[tele_i]->svx_data[most_hitch_cnt_R[tele_i]].hitch;

      if(most_hitch_cnt_F[tele_i] != -1 && most_hitch_cnt_R[tele_i] != -1)
	hitmap_svx[tele_i]
	  ->Fill(adc_max_ch_F[tele_i], adc_max_ch_R[tele_i]-256);
      
      // std::cout << " fill end (except correlation plot) - telescope No."
      // << tele_i << std::endl;
    }

    ////////////////////////////////////
    // make correlation plots
    for(int tele_i=0; tele_i<tele_num; tele_i++){
      for(int tele_j=tele_i; tele_j<tele_num; tele_j++){
      
	if(most_hitch_cnt_F[tele_i] != -1
	   && most_hitch_cnt_F[tele_j] != -1)
	  correlation_svx_Front[tele_i][tele_j]
	    ->Fill(adc_max_ch_F[tele_i], adc_max_ch_F[tele_j]);
      
	if(most_hitch_cnt_R[tele_i] != -1
	   && most_hitch_cnt_R[tele_j] != -1)
	  correlation_svx_Rear[tele_i][tele_j]
	    ->Fill(adc_max_ch_R[tele_i]-256, adc_max_ch_R[tele_j]-256);
    
      }
    }

    std::cout << " fill end - event_number : "
	      << ef_header_ana.event_number << std::endl;
  
    return ev_num;
  }

  void Analyze::HitPosition(const char *data, Int_t *hitx, Int_t *hity){
    unpack_data(data);

    std::cout<<"Fill start"<<std::endl;
    rest_data_exist = true;

    int tele_num = ef_header_ana.ef_number;
    int ev_num = ef_header_ana.event_number;

    int most_hitch_cnt_F[TELESCOPE_NUM] = {0};
    int most_hitch_cnt_R[TELESCOPE_NUM] = {0};

    int adc_max_ch_F[TELESCOPE_NUM] = {0};
    int adc_max_ch_R[TELESCOPE_NUM] = {0};
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      
      fill_adc_vs_ch(tele_i, most_hitch_cnt_F[tele_i], most_hitch_cnt_R[tele_i]);
    
      if(most_hitch_cnt_F[tele_i] != -1)
	adc_max_ch_F[tele_i]
	  = (int)svx_ef_ana[tele_i]->svx_data[most_hitch_cnt_F[tele_i]].hitch;
      
      if(most_hitch_cnt_R[tele_i] != -1)
	adc_max_ch_R[tele_i]
	  = (int)svx_ef_ana[tele_i]->svx_data[most_hitch_cnt_R[tele_i]].hitch;
    }

    for(int tele_i=0;tele_i<TELESCOPE_NUM;tele_i++){
      //std::cout << "hitx " << adc_max_ch_F[tele_i] << " hity " << adc_max_ch_R[tele_i]-256 << std::endl;
      *hitx = adc_max_ch_F[tele_i];
      *hity = adc_max_ch_R[tele_i]-256;
      hitx++;
      hity++;
    }
  }


  void Analyze::write_rootfile(){

    TFile* root_out = new TFile(filename.c_str(), "recreate");
  
    // rfile->cd();
    std::cout << " writing root file..." << std::endl;

    hist_time_stamp->Write();
    SVX4_TSvsEv->Write();
  
    for(int i=0;i<TELESCOPE_NUM;i++){
      Meanch_svx_Front[i] = hist_svx_Front[i]->ProfileX();
      Meanch_svx_Rear[i] = hist_svx_Rear[i]->ProfileX();
      Meanch_svx_Front[i]->SetOption("HistE");
      Meanch_svx_Front[i]->Write();
      Meanch_svx_Rear[i]->SetOption("HistE");
      Meanch_svx_Rear[i]->Write();
      hist_svx_Front[i]->SetOption("colz");
      hist_svx_Front[i]->Write();
      hist_svx_Rear[i]->SetOption("colz");
      hist_svx_Rear[i]->Write();
      hitmap_svx[i]->SetOption("colz");
      hitmap_svx[i]->Write();
      hist_pipelineID_svx_Front[i]->SetOption("colz");
      hist_pipelineID_svx_Front[i]->Write();
      hist_pipelineID_svx_Rear[i]->SetOption("colz");
      hist_pipelineID_svx_Rear[i]->Write();
      event_valid[i]->Write();
    
      for(int j=i; j<TELESCOPE_NUM; j++){
	correlation_svx_Front[i][j]->SetOption("colz");
	correlation_svx_Front[i][j]->Write();
	correlation_svx_Rear[i][j]->SetOption("colz");
	correlation_svx_Rear[i][j]->Write();
      }
    }

    rest_data_exist = false;
    root_out->Close();

    delete root_out;
  }


  void Analyze::get_config_param_ana(SVXconfig* config){
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      for(int chip_i=0; chip_i<4; chip_i++){
	for(int i=0; i<16; i++)
	  mask_config[tele_i][chip_i][i] = (int)config->getparam((param_names)i, tele_i, chip_i);
	disable_config[tele_i][chip_i] = (int)config->getparam(Disable, tele_i, chip_i);
	chipID_config[tele_i][chip_i] = (int)config->getparam(ID, tele_i, chip_i);
	thresh_config[tele_i][chip_i] = (int)config->getparam(Thresh, tele_i, chip_i);
      
	for(int j=0; j<16; j++){
	  std::string tmp_str = uc2binary((unsigned char)mask_config[tele_i][chip_i][j]);
	  for(int k=0; k<8; k++){
	    if(tmp_str[k] == '1'
	       && !masked_ch[tele_i][chip_i*128 + (15-j)*8 + (7-k)]){
	      // set mask flag and fill mask pattern
	      masked_ch[tele_i][chip_i*128 + (15-j)*8 + (7-k)] = true;
	      if(m_test_mode == Mask){
		for(int n=0; n<256; n++){
		  if(chip_i == 0 || chip_i == 1)
		    mask_pattern[tele_i]->Fill(chip_i*128 + (15-j)*8 + (7-k), n);
		  else if(chip_i == 3 || chip_i == 2)
		    mask_pattern[tele_i]->Fill(n, (chip_i-2)*128 + (15-j)*8 + (7-k));
		}
	      }
	    }
	  }
	}
      }
    }
    
  }


  void Analyze::Reset_hists(){
    hist_time_stamp->Reset();
    SVX4_TSvsEv->Reset();
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      hist_svx_Front[tele_i]->Reset();
      hist_svx_Rear[tele_i]->Reset();
      hitmap_svx[tele_i]->Reset();
      hist_pipelineID_svx_Front[tele_i]->Reset();
      hist_pipelineID_svx_Rear[tele_i]->Reset();
      event_valid[tele_i]->Reset();
      for(int tele_j=tele_i; tele_j<TELESCOPE_NUM; tele_j++){
	correlation_svx_Front[tele_i][tele_j]->Reset();
	correlation_svx_Rear[tele_i][tele_j]->Reset();
      }
    }
  }


  ////////////////////////////////////////////////////////////////////
  // Methods For Test Mode
  ////////////////////////////////////////////////////////////////////
  int Analyze::analyze_test_result(Run_Mode run_mode, int iteration, 
				   std::vector <struct change_param> &change_params){
    int result = -1;

    switch(m_test_mode){
    case Mask:
      result = analyze_dead_ch(run_mode, iteration, change_params);
      break;
    case Gain:
      result = SetPoint_gain_curve(run_mode, iteration);
      break;
    case Threshold:
      result = analyze_threshold(run_mode, iteration, change_params);
      break;
    case Pickdel_scan:
      result = analyze_hitnum_vs_pickdel(run_mode, iteration, change_params);
      break;
    case Trig_delay:
      result = analyze_hitnum_vs_delay(run_mode, iteration);
      break;
    default:
      // Do Noting
      break;
    }
  
    return result;
  }


  void Analyze::write_test_result(){

#ifdef SCTJDAQ_COMPILE
    std::string filepath = "/home/sctjdaq/data/";
#else
    std::string filepath = "data/";
#endif
    
    switch(m_test_mode){
    case Mask:
      write_mask_pattern(filepath + "Mask/mask.root");
      break;
    case Gain:
      write_gain_curve(filepath + "Gain/gain_curve.root");
      break;
    case Threshold:
      write_threshold_data(filepath + "Threshold/threshold.root");
      break;
    case Pickdel_scan:
      write_pickdel_scan_result(filepath + "Pickdel_scan/pickdel_scan.root");
      break;
    case Trig_delay:
      write_delay_scan_result(filepath + "Trig_delay/delay_scan.root");
      break;
    default:
      // Do Noting
      break;
    }
  
  }


  void Analyze::show_message(int iteration){
    
    switch(m_test_mode){
    case Mask:
      std::cout << "=============== Mask Scan ===============" << std::endl
		<< "Press Any Key." << std::endl;
      getchar();
      std::cout << "Measurement Start." << std::endl;
      break;
    case Gain:
      std::cout << "=============== Gain Measurement ===============" << std::endl;
      std::cout << "Set VCAL pad to " << (double)iteration*0.5 << "V, "
		<< "and Press Any Key." << std::endl;
      test_value = (Double_t)iteration * 0.5;
      getchar();
      std::cout << "Measurement Start." << std::endl;
      break;
    case Threshold:
      std::cout << "=============== Threshold Scan ===============" << std::endl
		<< "Press Any Key." << std::endl;
      getchar();
      std::cout << "Measurement Start." << std::endl;
      break;
    case Pickdel_scan:
      std::cout << "=============== PickDel Scan ===============" << std::endl;
      test_value = iteration + 3;
      break;
    case Trig_delay:
      std::cout << "=============== Trigger Delay Scan ===============" << std::endl;
      std::cout << "Set delay width to " << (double)iteration*10 << "ns, "
		<< "and Press Any Key." << std::endl;
      test_value = (Double_t)iteration * 10;
      getchar();
      std::cout << "Measuremenr Start." << std::endl;
      break;
    default:
      std::cout << "=============== DAQ mode ===============" << std::endl
		<< "Press Any Key." << std::endl;
      getchar();
      std::cout << "DAQ Start." << std::endl;
      break;
    }
  
  }


  void Analyze::init_conf_for_test(SVXconfig* config){
    
    switch(m_test_mode){
    case Mask:
      config->set_allMask(0);
      config->set_pedestal_taking();
      config->set_RdAll();
      break;
    case Gain:
      config->set_Qinj_taking();
      config->set_RdAll();
      break;
    case Threshold:
      config->set_pedestal_taking();
      config->set_RdAll();
      break;
    case Pickdel_scan:
      config->writeparam_allchips(PickDel, 3/*Not chrage injected cell*/);
      config->set_sparsification();
      break;
    case Trig_delay:
      config->set_sparsification();
      break;
    default:
      // Do Noting
      break;
    }
  
  }


  bool Analyze::whether_rewritten(param_names param, Test_Mode test_mode){
    
    switch(test_mode){
    case Mask:
      if(param >= 0 && param < 16) return true;
      break;
    case Gain:
      return false;
      break;
    case Threshold:
      if(param == Thresh) return true;
      break;
    case Pickdel_scan:
      if(param == PickDel) return true;
      break;
    case Trig_delay:
      return false;
      break;
    default:
      break;
    }

    return false; // default
  }


  int Analyze::rewrite_config_file(SVXconfig* config){

    std::string conf_path;
    std::string conf_file;
#ifdef SCTJDAQ_COMPILE
    conf_path = "/home/sctjdaq/src/telescope/text/";
#else
    conf_path = "../text/";
#endif
    
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      
      char filename[20];
      sprintf(filename, "config_tele%d.dat", tele_i+1);
      conf_file = filename;

      std::ifstream inf((conf_path + conf_file).c_str());
      if(!inf){
	std::cout << "Error! Failed to Open File. ("
		  << conf_path + conf_file <<')' << std::endl;
	return -1;
      }
      
      std::ofstream conff((conf_path + "conf_new/" + conf_file).c_str(),
			  std::ios::trunc);
      if(!conff){
	std::cout << "Error! Failed to Open File. ("
		  << conf_path + "conf_new/" + conf_file <<')' << std::endl;
	return -1;
      }

      // config rewritten methods
      std::string lineStr, tmpStr;
      unsigned char val;
      int chip_i;

      while(!inf.eof()){
	getline(inf, lineStr);
	tmpStr = lineStr;

	if(tmpStr.find_first_of('[') != std::string::npos)
	  tmpStr.erase(tmpStr.find_first_of('[')); // Trim the end of tmpStr
	if(tmpStr.find_first_of('\t') != std::string::npos)
	  tmpStr.erase(tmpStr.find_first_of('\t')); // Trim the end of tmpStr
	tmpStr += " ";

	if(tmpStr.compare(0, 9, "#chip_no.", 9) == 0){
	  chip_i = atoi(lineStr.c_str() + lineStr.size() - 2) - (tele_i*4) -1;
	}else{
	  for(int i=0; i<PARAM_INDEX_END; i++){
	    param_names param = (param_names)i;
	    int param_digit = config->digit(param);
	    
	    if(strncmp(Strparam_names(param).c_str(),
		       tmpStr.c_str(),
		       Strparam_names((param_names)i).find_first_of(" ") + 1) == 0){
		       // tmpStr.c_str(), config->name_length(param) + 1) == 0){

	      if(whether_rewritten(param, m_test_mode)){
		val = config->getparam(param, tele_i, chip_i, 0);

		if(config->gray(param))
		  tmpStr = uc2binary(gray_encoder(val));
		else
		  tmpStr = uc2binary(val);

		if(config->direction(param)){
		  lineStr.replace(lineStr.size() - param_digit, param_digit,
				  tmpStr, 8 - param_digit, param_digit);
		}else{
		  tmpStr = inv_str(tmpStr.c_str());
		  lineStr.replace(lineStr.size() - param_digit, param_digit,
				  tmpStr, 0, param_digit);
		}
	      }
	      
	      // std::cout << Strparam_names(param) << tmpStr << std::endl; // For Debug
	      // std::cout << lineStr << std::endl; // For Debug
	      break;
	    }
	  }
	}
	
	conff << lineStr << std::endl;
      }

      inf.close();
      conff.close();
    }

    return 0;
  }


  ///////////////////////////////////////////////
  // Mask Scan
  ///////////////////////////////////////////////
  // To be improved
  int Analyze::analyze_dead_ch(Run_Mode run_mode, int iteration,
			       std::vector <struct change_param> &change_params){
    
    int prev_dead_ch_num = dead_ch_num;

    std::cout << "Iteration " << iteration << std::endl;// debug
  
    if(iteration >= 5)
      return TEST_END;
    else if(run_mode == DATA)
      return TEST_END;

    Double_t Mean_F[TELESCOPE_NUM] = {0};
    Double_t Mean_R[TELESCOPE_NUM] = {0};
    Double_t Maxbin_mean_F[TELESCOPE_NUM] = {0};
    Double_t Maxbin_mean_R[TELESCOPE_NUM] = {0};
    Double_t Sigma_mean_F[TELESCOPE_NUM] = {0};
    Double_t Sigma_mean_R[TELESCOPE_NUM] = {0};
  
    // Caluculate mean of each channels mean and RMS
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      int ch_cnt_F = 0;
      int ch_cnt_R = 0;
        
      for(int ch=0; ch<256; ch++){
	TH1D* hist_F = hist_svx_Front[tele_i]->ProjectionY("hist_F_pjy", ch+1, ch+1);
	TH1D* hist_R = hist_svx_Rear[tele_i]->ProjectionY("hist_R_pjy", ch+1, ch+1);
	if(!masked_ch[tele_i][ch]){
	  Mean_F[tele_i] += hist_F->GetMean(1);
	  Sigma_mean_F[tele_i] += hist_F->GetRMS(1);
	  if(iteration == 0)
	    Maxbin_mean_F[tele_i] += hist_F->GetBinContent(hist_F->GetMaximumBin());
	  else if(iteration == 2)
	    Maxbin_mean_F[tele_i] += hist_F->GetMaximumBin();
	  ch_cnt_F += 1.0;
	}
	if(!masked_ch[tele_i][ch+256]){
	  Mean_R[tele_i] += hist_R->GetMean(1);
	  Sigma_mean_R[tele_i] += hist_R->GetRMS(1);
	  if(iteration == 0)
	    Maxbin_mean_R[tele_i] += hist_R->GetBinContent(hist_R->GetMaximumBin());
	  else if(iteration == 2)
	    Maxbin_mean_R[tele_i] += hist_R->GetMaximumBin();
	  ch_cnt_R += 1.0;
	}
      }
    
      Mean_F[tele_i] = Mean_F[tele_i]/ch_cnt_F;
      Sigma_mean_F[tele_i] = Sigma_mean_F[tele_i]/ch_cnt_F;
      Maxbin_mean_F[tele_i] = Maxbin_mean_F[tele_i]/ch_cnt_F;
      Mean_R[tele_i] = Mean_R[tele_i]/ch_cnt_R;
      Sigma_mean_R[tele_i] = Sigma_mean_R[tele_i]/ch_cnt_R;
      Maxbin_mean_R[tele_i] = Maxbin_mean_R[tele_i]/ch_cnt_R;
      std::cout << Mean_F[tele_i] << " " << Sigma_mean_F[tele_i] << " " << Maxbin_mean_F[tele_i] << std::endl
		<< Mean_R[tele_i] << " " << Sigma_mean_R[tele_i] << " " << Maxbin_mean_R[tele_i] << std::endl; // debug
    }

    // Determine mask channel
    Double_t ch_Mean_F, ch_Mean_R, Maxbin_F, Maxbin_R;
  
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      for(int ch=0; ch<256; ch++){
	TH1D* hist_F = hist_svx_Front[tele_i]->ProjectionY("hist_F_pjy", ch+1, ch+1);
	TH1D* hist_R = hist_svx_Rear[tele_i]->ProjectionY("hist_R_pjy", ch+1, ch+1);
      
	prev_masked_ch[tele_i][ch] = masked_ch[tele_i][ch];
	prev_masked_ch[tele_i][ch+256] = masked_ch[tele_i][ch+256];
      
	ch_Mean_F = hist_F->GetMean(1);
	ch_Mean_R = hist_R->GetMean(1);

	if(iteration == 0){
	  Maxbin_F = hist_F->GetBinContent(hist_F->GetMaximumBin());
	  Maxbin_R = hist_R->GetBinContent(hist_R->GetMaximumBin());
	}else if(iteration == 2){
	  Maxbin_F = hist_F->GetMaximumBin();
	  Maxbin_R = hist_R->GetMaximumBin();
	}
	
      
	if(iteration == 0){
	  // To remove channel that has too small fluctuation of adc value
	  if(Maxbin_F > Maxbin_mean_F[tele_i] + num_event*0.3
	     && !masked_ch[tele_i][ch]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch] = true;
	    for(int i=0; i<256; i++)
	      mask_pattern[tele_i]->Fill(ch, i);
	  }
	  if(Maxbin_R > Maxbin_mean_R[tele_i] + num_event*0.3
	     && !masked_ch[tele_i][ch+256]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch+256] = true;
	    for(int i=0; i<256; i++)
	      mask_pattern[tele_i]->Fill(i, ch);
	  }
	}else if(iteration == 1){
	  // To remove channel that has too small fluctuation of adc value
	  if((hist_svx_Front[tele_i]->ProfileX())->GetMaximumBin() == ch+1
	     && !masked_ch[tele_i][ch]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch] = true;
	    for(int i=0; i<256; i++)
	      mask_pattern[tele_i]->Fill(ch, i);
	  }
	  if((hist_svx_Rear[tele_i]->ProfileX())->GetMaximumBin() == ch+1
	     && !masked_ch[tele_i][ch+256]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch+256] = true;
	    for(int i=0; i<256; i++)
	      mask_pattern[tele_i]->Fill(i, ch);
	  }
	}else if(iteration == 2){
	  // To remove channel that has too small fluctuation of adc value
	  if((Maxbin_F > Maxbin_mean_F[tele_i] + 1.5*Sigma_mean_F[tele_i]
	      || Maxbin_F < Maxbin_mean_F[tele_i] - 1.5*Sigma_mean_F[tele_i])
	     && !masked_ch[tele_i][ch]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch] = true;
	    for(int i=0; i<256; i++)
	      mask_pattern[tele_i]->Fill(ch, i);
	  }
	  if((Maxbin_R > Maxbin_mean_R[tele_i] + 1.2*Sigma_mean_R[tele_i]
	      || Maxbin_R < Maxbin_mean_R[tele_i] - 1.2*Sigma_mean_R[tele_i])
	     && !masked_ch[tele_i][ch+256]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch+256] = true;
	    for(int i=0; i<256; i++)
	      mask_pattern[tele_i]->Fill(i, ch);
	  }	
	}else{
	  // To remove channel that has very difference adc value from others
	  if((ch_Mean_F > Mean_F[tele_i] + 5*Sigma_mean_F[tele_i]
	      || ch_Mean_F < Mean_F[tele_i] - 5*Sigma_mean_F[tele_i])
	     && !masked_ch[tele_i][ch]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch] = true;
	    for(int i=0; i<256; i++)
	      mask_pattern[tele_i]->Fill(ch, i);
	  }
	  if((ch_Mean_R > Mean_R[tele_i] + 5*Sigma_mean_R[tele_i]
	      || ch_Mean_R < Mean_R[tele_i] - 5*Sigma_mean_R[tele_i])
	     && !masked_ch[tele_i][ch+256]){
	    dead_ch_num++;
	    masked_ch[tele_i][ch+256] = true;
	    for(int i=0; i<256; i++)	
	      mask_pattern[tele_i]->Fill(i, ch);
	  }
	}
      }
    
      set_mask_param(tele_i, change_params);
    }

    if(prev_dead_ch_num == dead_ch_num) return TEST_END;

    return dead_ch_num;
  }


  void Analyze::write_mask_pattern(std::string filename){
    TFile* root_mask = new TFile(filename.c_str(), "recreate");
    std::cout << " Writing Mask pettern..." << std::endl;

    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      mask_pattern[tele_i]->SetOption("colz");
      mask_pattern[tele_i]->Write();
    }
  
    root_mask->Close();
    delete root_mask;
  }

  
  void Analyze::set_mask_param(int tele_i, std::vector <struct change_param> &change_params){
    bool change_mask[4][16] = {{false}};

    // To get present mask parameter value
    int mask_param[4][16] = {{0}};
    for(int chip_i=0; chip_i<4; chip_i++){
      for(int mask_i=0; mask_i<16; mask_i++){
	mask_param[chip_i][mask_i] = mask_config[tele_i][chip_i][mask_i];
      }
    }

    // To set mask_param
    for(int ch_i=0; ch_i<128*4; ch_i++){
      int chip_i = ch_i/128;
      int mask_i = 15 - (ch_i%128)/8;
      if(!prev_masked_ch[tele_i][ch_i] && masked_ch[tele_i][ch_i]){
	mask_param[chip_i][mask_i]
	  = (int)((unsigned char)mask_param[chip_i][mask_i] | (unsigned char)pow(2, (ch_i%128)%8));
	change_mask[chip_i][mask_i] = true;
      }
    }

    // To fill change_params
    for(int chip_i=0; chip_i<4; chip_i++){
      for(int mask_i=0; mask_i<16; mask_i++){
	if(change_mask[chip_i][mask_i]){
	  change_param tmp_param = {(param_names)mask_i, mask_param[chip_i][mask_i], tele_i, chip_i};
	  change_params.push_back(tmp_param);
	}
      }
    }
  
  }


  ///////////////////////////////////////////////
  // Gain Measurement
  ///////////////////////////////////////////////
  int Analyze::SetPoint_gain_curve(Run_Mode run_mode, int point_num){

    Double_t volt = test_value;
    Mask_mode = false;
  
    if(point_num >= 5)           return TEST_END;
    else if(run_mode == DATA)    return 0;
  
    Double_t x_F, x_R, dx_F, dx_R;

    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      for(int num=0; num<32; num++){
	x_F = (hist_svx_Front[tele_i]->ProfileX())->GetBinContent(num*8+4 + 1);
	dx_F = (hist_svx_Front[tele_i]->ProfileX())->GetBinError(num*8+4 + 1);
	x_R = (hist_svx_Rear[tele_i]->ProfileX())->GetBinContent(num*8+4 + 1);
	dx_R = (hist_svx_Rear[tele_i]->ProfileX())->GetBinError(num*8+4 + 1);
    
	gain_curve_Front[tele_i][num]->SetPoint(point_num, volt, x_F);
	gain_curve_Front[tele_i][num]->SetPointError(point_num, 0, dx_F);
	gain_curve_Rear[tele_i][num]->SetPoint(point_num, volt, x_R);
	gain_curve_Rear[tele_i][num]->SetPointError(point_num, 0, dx_R);
      }
    }

    return point_num;
  }


  void Analyze::write_gain_curve(std::string filename){
  
    TFile* root_gain = new TFile(filename.c_str(), "recreate");
    std::cout << " Writing Gain curve..." << std::endl;

    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      for(int num=0; num<32; num++){
	gain_curve_Front[tele_i][num]->Write();
	gain_curve_Rear[tele_i][num]->Write();
      }
    }
  
    root_gain->Close();
    delete root_gain;
  }


  ///////////////////////////////////////////////
  // Threshold Scan
  ///////////////////////////////////////////////
  int Analyze::analyze_threshold(Run_Mode run_mode, int iteration,
			std::vector <struct change_param> &change_params){

    if(run_mode == DATA)  return TEST_END;

    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      
      for(int chip_i=0; chip_i<4; chip_i++){
	double Mean_Mean = 0;
	double Sigma_Mean = 0;
	int ch_count = 0;
	
	for(int ch=0; ch<128; ch++){
	  
	  if(chip_i == 0 || chip_i == 1){
	    int bin_ch = ch + chip_i*128 + 1;
	    TH1D* hist_F = hist_svx_Front[tele_i]->ProjectionY("", bin_ch, bin_ch);
	    pedestal_mean_Front[tele_i]->SetBinContent(bin_ch, hist_F->GetMean(1));
	    pedestal_mean_Front[tele_i]->SetBinError(bin_ch, hist_F->GetRMS(1));

	    if(!masked_ch[tele_i][bin_ch - 1]){
	      Mean_Mean += hist_F->GetMean(1);
	      Sigma_Mean += hist_F->GetRMS(1);
	      ch_count++;
	    }
	    
	  }else if(chip_i == 2 || chip_i == 3){
	    int bin_ch = ch + (chip_i-2)*128 + 1;
	    TH1D* hist_R = hist_svx_Rear[tele_i]->ProjectionY("", bin_ch, bin_ch);
	    pedestal_mean_Rear[tele_i]->SetBinContent(bin_ch, hist_R->GetMean(1));
	    pedestal_mean_Rear[tele_i]->SetBinError(bin_ch, hist_R->GetRMS(1));

	    if(!masked_ch[tele_i][bin_ch+256-1]){
	      Mean_Mean += hist_R->GetMean(1);
	      Sigma_Mean += hist_R->GetRMS(1);
	      ch_count++;
	    }
	  }
	  
	}
	  
	Mean_Mean = Mean_Mean/ch_count;
	Sigma_Mean = Sigma_Mean/ch_count;

	change_param tmp_param =
	  {Thresh, (int)(Mean_Mean + Sigma_Mean*3), tele_i, chip_i};
	change_params.push_back(tmp_param);
      }
    }

    return TEST_END; // No Iteration
  }

  void Analyze::write_threshold_data(std::string filename){

    TFile* root_thresh = new TFile(filename.c_str(), "recreate");
    std::cout << " Writing Threshold Data..." << std::endl;
    
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      pedestal_mean_Front[tele_i]->Write();
      pedestal_mean_Rear[tele_i]->Write();
    }
    
    root_thresh->Close();
    delete root_thresh;
  }

  
  ///////////////////////////////////////////////
  // PickDel Scan
  ///////////////////////////////////////////////
  int Analyze::analyze_hitnum_vs_pickdel(Run_Mode run_mode, int point_num,
					 std::vector <struct change_param> &change_params){
    
    if(point_num >= 30)          return TEST_END;
    else if(run_mode == DATA)    return 0;

    int pickdel = (int)test_value;
    int hit_num = 0;
    
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++){
      hit_num += hitmap_svx[tele_i]->Integral(0, 256, 0, 256);
      for(int chip_i=0; chip_i<4; chip_i++){
	change_param tmp_param = {PickDel, pickdel+1, tele_i, chip_i};
	change_params.push_back(tmp_param);
      }
    }
    
    hitnum_vs_pickdel->SetPoint(point_num, pickdel, hit_num);

    std::cout << "Hit Number : " << hit_num << std::endl;
    return hit_num;
  }
  
  void Analyze::write_pickdel_scan_result(std::string filename){
    
    TFile* root_pickdel = new TFile(filename.c_str(), "recreate");
    std::cout << " Writing Pickdel scan result..." << std::endl;
    
    hitnum_vs_pickdel->Write();
    
    root_pickdel->Close();
    delete root_pickdel;
  }

  ///////////////////////////////////////////////
  // Delay Scan
  ///////////////////////////////////////////////
  int Analyze::analyze_hitnum_vs_delay(Run_Mode run_mode, int point_num){

    if(point_num >= 10)          return TEST_END;
    else if(run_mode == DATA)    return 0;

    int delay = point_num;
    int hit_num = 0;
    
    for(int tele_i=0; tele_i<TELESCOPE_NUM; tele_i++)
      hit_num += hitmap_svx[tele_i]->Integral(0, 256, 0, 256);
    
    hitnum_vs_delay->SetPoint(point_num, delay, hit_num);

    std::cout << "Hit Number : " << hit_num << std::endl;
    return hit_num;
  }
  
  void Analyze::write_delay_scan_result(std::string filename){
    
    TFile* root_delay = new TFile(filename.c_str(), "recreate");
    std::cout << " Writing Delay scan result..." << std::endl;
    
    hitnum_vs_delay->Write();
    
    root_delay->Close();
    delete root_delay;
  }

} // namespace svx
