module mode_changer(
		    OPERATE_CLK_IN,
		    ACTIVE_IN,
		    RESET_IN,
		    BEMODE_IN,
		    FEMODE_IN,
		    RUN_FLAG_IN,
		    BEMODE_OUT,
		    FEMODE_OUT,
		    CHMODE_OUT,
		    INIT_FLAG,
		    ACQ_FLAG,
		    DIGIT_FLAG,
		    READ_FLAG,
		    RST_FLAG,
		    ERROR_OUT
		    );

   
   input            OPERATE_CLK_IN; // Main Operational Clock 
   input 	    ACTIVE_IN;	    // Active flag
   input 	    RESET_IN;	    // System RESET
   
   input 	    BEMODE_IN;	    // ////////////////////// 
   input 	    FEMODE_IN;	    // from ctrl_sig_maker
   input 	    RUN_FLAG_IN;    // DAQ is RUNNING sign
   
   output reg	    BEMODE_OUT;	    // to SVX4 BEMODE pad
   output reg	    FEMODE_OUT;	    // to SVX4 FEMODE pad
   output reg	    CHMODE_OUT;	    // to SVX4 CHMODE pad
   
   output reg 	    INIT_FLAG;	    // /////////////////
   output reg	    ACQ_FLAG;	    // /////////////////
   output reg	    DIGIT_FLAG;	    // mode indicators
   output reg	    READ_FLAG;	    // /////////////////
   output reg 	    RST_FLAG;	    // /////////////////
   
   output reg 	    ERROR_OUT;	    // Error

   reg  	    bemode_r;
   reg 		    femode_r;
   
   reg 		    invalid_change_r;

   reg [3:0] 	    chmode_counter_r;

   reg 		    hold_bemode_r;
   reg 		    hold_femode_r;
   reg [3:0] 	    hold_counter_r;
   
   reg		    mode_changing_r;
   reg 		    mode_changed_r;
   
   reg 		    activated_r;
   reg [7:0] 	    activate_cnt_r;

   parameter [3:0] require_holdtime = 4'd2;
   parameter [3:0] chmode_width = 4'd4;
   

   ///////////////////////////////////////
   // Initialize and Terminate of Run
   //
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 activated_r <= 1'b0;
	 activate_cnt_r <= 8'd0;
	 bemode_r <= 1'b1;
	 femode_r <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    if(activated_r == 1'b0)begin
	       // completely terminated
	       activate_cnt_r <= 8'd0;
	       bemode_r <= 1'b1;
	       femode_r <= 1'b0;
	    end else begin
	       // Termination Method
	       if(activate_cnt_r[7:0] < 8'd12)begin
		  // To avoid invalid transition
		  activate_cnt_r[7:0] <= activate_cnt_r[7:0] + 8'd1;
	       end else if(activate_cnt_r[7:0] < 8'd30) begin
		  activate_cnt_r[7:0] <= activate_cnt_r[7:0] + 8'd1;
		  bemode_r <= 1'b1;
		  femode_r <= 1'b0;
	       end else begin
		  activated_r <= 1'b0;
	       end
	    end
   	 end else begin
	    if(activated_r == 1'b0)begin
	       // Not Yet Activated
	       if(activate_cnt_r[7:0] < 8'd4)begin
		  // To avoid invalid transition
		  activate_cnt_r[7:0] <= activate_cnt_r[7:0] + 8'd1;
	       end else if(activate_cnt_r[7:0] < 8'd20) begin
		  activate_cnt_r[7:0] <= activate_cnt_r[7:0] + 8'd1;
		  bemode_r <= 1'b1;
		  femode_r <= 1'b0;
	       end else begin
		  activated_r <= 1'b1;
	       end
	    end else begin
	       // Activated
	       activate_cnt_r <= 8'd0;
	       bemode_r <= BEMODE_IN;
	       femode_r <= FEMODE_IN;
	    end
   	 end
      end
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   

   /////////////////////////////////////
   // Discriminate By Duration
   //
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 hold_bemode_r <= 1'b1;
	 hold_femode_r <= 1'b0;
	 mode_changing_r <= 1'b0;
	 hold_counter_r[3:0] <= 4'd0;
	 invalid_change_r <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)begin
	    hold_bemode_r <= 1'b1;
	    hold_femode_r <= 1'b0;
	    mode_changing_r <= 1'b0;
	    hold_counter_r[3:0] <= 4'd0;
	    invalid_change_r <= 1'b0;
   	 end else begin
	    
	    if(hold_bemode_r == bemode_r && hold_femode_r == femode_r)begin
	       if(mode_changed_r == 1'b1)begin
		  mode_changing_r <= 1'b0;	// Completely mode changed
	       end
	    end else begin
	       // BEMODE_IN or FEMODE_IN are changing
	       if(hold_counter_r[3:0] >= require_holdtime)begin
		  if(mode_changing_r == 1'b0)begin
		     hold_bemode_r <= bemode_r;
		     hold_femode_r <= femode_r;
		     mode_changing_r <= 1'b1;
		     hold_counter_r[3:0] <= 4'd0;
		  end else begin
		     invalid_change_r <= 1'b1;
		  end
	       end else begin
		  hold_counter_r[3:0] <= hold_counter_r[3:0] + 4'd1;
	       end // else: !if(hold_counter_r[3:0] >= require_holdtime)
	    end // else: !if(hold_bemode_r == bemode_r...
	    
   	 end // else: !if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   

   /////////////////////////////////////
   // Mode Changer
   //
   always @(negedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 BEMODE_OUT <= 1'b1;
	 FEMODE_OUT <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)begin
	    BEMODE_OUT <= 1'b1;
	    FEMODE_OUT <= 1'b0;
   	 end else begin
	    
	    if(chmode_counter_r[3:0] == (chmode_width/2 + 4'd1))begin
	       if(hold_bemode_r == 1'b0 && hold_femode_r == 1'b0)begin
		  BEMODE_OUT <= 1'b0;
		  FEMODE_OUT <= 1'b0;
	       end else if(hold_bemode_r == 1'b1 && hold_femode_r == 1'b0)begin
		  BEMODE_OUT <= 1'b1;
		  FEMODE_OUT <= 1'b0;
	       end else if(hold_bemode_r == 1'b1 && hold_femode_r == 1'b1)begin
		  BEMODE_OUT <= 1'b1;
		  FEMODE_OUT <= 1'b1;
	       end else if(hold_bemode_r == 1'b0 && hold_femode_r == 1'b1)begin
		  BEMODE_OUT <= 1'b0;
		  FEMODE_OUT <= 1'b1;
	       end
	    end else begin
	       BEMODE_OUT <= BEMODE_OUT;
	       FEMODE_OUT <= FEMODE_OUT;
	    end // else: !if(chmode_counter_r[3:0] == (chmode_width/2 + 4'd1))
	    
   	 end // else: !if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 CHMODE_OUT <= 1'b0;
      end else begin
	 if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)begin
	    CHMODE_OUT <= 1'b0;
	 end else begin
	    
	    if(mode_changing_r == 1'b1 && mode_changed_r == 1'b0)begin
	       CHMODE_OUT <= 1'b1;
	    end else begin
	       CHMODE_OUT <= 1'b0;
	    end
	    
	 end // else: !if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN)
   
   
   /////////////////////////////////////
   // Mode Changer
   //   
   always @(posedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 chmode_counter_r[3:0] <= 4'd0;
	 mode_changed_r <= 1'b0;
      end else begin
	 if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)begin
	    chmode_counter_r[3:0] <= 4'd0;
	    mode_changed_r <= 1'b0;
	 end else begin
	    
	    if(mode_changing_r == 1'b1)begin
	       if(chmode_counter_r[3:0] < chmode_width)begin
		  chmode_counter_r[3:0] <= chmode_counter_r[3:0] + 4'd1;
	       end else if(chmode_counter_r[3:0] == chmode_width)begin
		  mode_changed_r <= 1'b1;
	       end
	    end else begin
	       chmode_counter_r[3:0] <= 4'd0;
	       mode_changed_r <= 1'b0;
	    end
	    
	 end // else: !if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
   
   ///////////////////////////////
   // Mode Flags
   //
   always @(negedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 // Initial state
      	 INIT_FLAG <= 1'b0;
	 ACQ_FLAG <= 1'b1;
	 DIGIT_FLAG <= 1'b0;
	 READ_FLAG <= 1'b0;
	 RST_FLAG <= 1'b0;
      end else begin
	 if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)begin
	    // Initial state
      	    INIT_FLAG <= 1'b0;
	    ACQ_FLAG <= 1'b1;
	    DIGIT_FLAG <= 1'b0;
	    READ_FLAG <= 1'b0;
	    RST_FLAG <= 1'b0;
	 end else begin
	    
	    if(chmode_counter_r[3:0] == (chmode_width/2 + 4'd1))begin
	       if(RUN_FLAG_IN == 1'b0
		  && hold_bemode_r == 1'b0 && hold_femode_r == 1'b0)begin
		  INIT_FLAG <= 1'b1;
		  ACQ_FLAG <= 1'b0;
		  DIGIT_FLAG <= 1'b0;
		  READ_FLAG <= 1'b0;
		  RST_FLAG <= 1'b0;
	       end else if(hold_bemode_r == 1'b1 && hold_femode_r == 1'b0)begin
		  INIT_FLAG <= 1'b0;
		  ACQ_FLAG <= 1'b1;
		  DIGIT_FLAG <= 1'b0;
		  READ_FLAG <= 1'b0;
		  RST_FLAG <= 1'b0;
	       end else if(hold_bemode_r == 1'b1 && hold_femode_r == 1'b1)begin
		  INIT_FLAG <= 1'b0;
		  ACQ_FLAG <= 1'b0;
		  DIGIT_FLAG <= 1'b1;
		  READ_FLAG <= 1'b0;
		  RST_FLAG <= 1'b0;
	       end else if(hold_bemode_r == 1'b0 && hold_femode_r == 1'b1)begin
		  INIT_FLAG <= 1'b0;
		  ACQ_FLAG <= 1'b0;
		  DIGIT_FLAG <= 1'b0;
		  READ_FLAG <= 1'b1;
		  RST_FLAG <= 1'b0;
	       end else if(RUN_FLAG_IN == 1'b1
			   && hold_bemode_r == 1'b0 && hold_femode_r == 1'b0)begin
		  INIT_FLAG <= 1'b0;
		  ACQ_FLAG <= 1'b0;
		  DIGIT_FLAG <= 1'b0;
		  READ_FLAG <= 1'b0;
		  RST_FLAG <= 1'b1;
	       end
	    end // if (chmode_counter_r[3:0] == (chmode_width/2 + 4'd1))
	 end // else: !if(ACTIVE_IN == 1'b0 && activated_r == 1'b0)
      end // else: !if(RESET_IN == 1'b1)
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)


   //////////////////////////////////////
   // ERROR
   //
   always @(negedge OPERATE_CLK_IN or posedge RESET_IN)begin
      if(RESET_IN == 1'b1)begin
	 ERROR_OUT <= 1'b0;
      end else begin
   	 if(ACTIVE_IN == 1'b0)begin
	    ERROR_OUT <= 1'b0;
   	 end else begin
	    ERROR_OUT <=  invalid_change_r;
   	 end
      end
   end // always @ (posedge OPERATE_CLK_IN or posedge RESET_IN)
   
endmodule
