/*******************************************************************************
*                                                                              *
* System      : SiTCP                                                          *
* Block       : SIO                                                            *
* Module      : SIO_REG                                                        *
* Version     : v 0.0.0 2008/03/26 12:01                                       *
*                                                                              *
* Description : A sample for the Serial IO interface                           *
*                                                                              *
* Designer    : Tomohisa Uchida & Modified by Minoru Hirose                    *
*                                                                              *
*                Copyright (c) 2008 Tomohisa Uchida                            *
*                All rights reserved                                           *
*                                                                              *
*******************************************************************************/
module SIO_REG(
	       SCK     ,// in  : Clock
	       RSTn    ,// in  : System reset

	       // Register I/F
	       REG_ADDR,// in  : Address[31:0]
	       REG_WD  ,// in  : Data[7:0]
	       REG_WE  ,// in  : Write enable
	       REG_RE  ,// in  : Read enable
	       REG_ACK ,// out : Access acknowledge
	       REG_RV  ,// out : Read valid
	       REG_RD  ,// out : Read data[7:0]
	       CONF_PARAM,
	       USE_TELE,
	       TRIG_DELAY,
	       LED_SW,
	       SYS_VER  // in  : System Version info
	       );
   
   //-------- Input/Output -------------
   input			SCK			;
   input			RSTn			;
   
   input [31:0] 		REG_ADDR		;
   input [7:0] 			REG_WD			;
   input			REG_WE			;
   input			REG_RE			;
   input [31:0]			SYS_VER			;
   
   output			REG_ACK			;
   output			REG_RV			;
   output [7:0] 		REG_RD			;
   output [22:0] 		CONF_PARAM		;
   output [3:0] 		USE_TELE                ;
   output [3:0] 		TRIG_DELAY              ;
   output 			LED_SW                  ;
   
   //---------- Buffer ----------
   wire 			REG_ACK			;
   wire 			REG_RV			;
   wire [7:0] 			REG_RD			;
   
   //------------------------------------------------------------------------------
   //	Receive
   //------------------------------------------------------------------------------
   wire 			regCs			;

   
   // REGISER Space is from 0x100 to 0x1ff
   assign regCs = ~(|REG_ADDR[31:9]) & REG_ADDR[8] & REG_WE;


   //------------------------------------------------------------------------------
   //   Memory
   //------------------------------------------------------------------------------
   reg [2:0] 			Pipe_scan_r;
   reg [5:0] 			PickDel_r;
   reg [7:0] 			Latency_r;
   reg [7:0] 			Mask_len_r;
   assign                       CONF_PARAM[22:0] = {Pipe_scan_r[0], PickDel_r[5:0],
						    Latency_r[7:0], Mask_len_r[7:0]};

   reg [7:0] 			use_tele_r;
   assign                       USE_TELE[3:0] = use_tele_r[3:0];

   reg [7:0] 			trig_delay_r;
   assign                       TRIG_DELAY[3:0] = trig_delay_r[3:0];

   reg [7:0] 			led_sw_r;
   assign                       LED_SW = (led_sw_r[7:0] == 8'd0) ? 1'b1 : 1'b0;
   
   always @(posedge SCK)begin
      if(~RSTn)begin
	 use_tele_r[7:0] <= 8'b0000_0000;
	 trig_delay_r[7:0] <= 8'b0000_0000;
	 Mask_len_r[7:0] <= 8'b000_0000;
	 Latency_r[7:0] <= 8'b0000_0000;
	 {Pipe_scan_r[2:0], PickDel_r[5:0]} <= 8'b0000_0000;
	 led_sw_r[7:0] <= 8'b0000_0000;
      end else begin
	 if(REG_WE == 1'b1)begin
	    {Pipe_scan_r[2:0], PickDel_r[5:0]} <= (regCs & (REG_ADDR[7:0] == 8'h05)) ? REG_WD[7:0] : {Pipe_scan_r[2:0], PickDel_r[5:0]};	    
	    Latency_r[7:0]    <= (regCs & (REG_ADDR[7:0] == 8'h06)) ? REG_WD[7:0] : Latency_r[7:0];
	    Mask_len_r[7:0]   <= (regCs & (REG_ADDR[7:0] == 8'h07)) ? REG_WD[7:0] : Mask_len_r[7:0];
	    use_tele_r[7:0]   <= (regCs & (REG_ADDR[7:0] == 8'h08)) ? REG_WD[7:0] : use_tele_r[7:0];
	    led_sw_r[7:0]     <= (regCs & (REG_ADDR[7:0] == 8'h09)) ? REG_WD[7:0] : led_sw_r[7:0];
	    trig_delay_r[7:0] <= (regCs & (REG_ADDR[7:0] == 8'h0B)) ? REG_WD[7:0] : trig_delay_r[7:0];
	 end
      end
   end

   //Buffering wire signals
   reg		[7:0]	rdData		;
   reg 			orRv		;
   reg 			orAck		;
   
   always @(posedge SCK)begin
      case(REG_ADDR[7:0])	
	8'h00:	rdData[7:0]	<= SYS_VER[7:0];
	8'h01:	rdData[7:0]	<= SYS_VER[15:8];
	8'h02:	rdData[7:0]	<= SYS_VER[23:16];
	8'h03:	rdData[7:0]	<= SYS_VER[31:24];
	8'h05:	rdData[7:0]	<= {Pipe_scan_r[2:0], PickDel_r[5:0]};
	8'h06:	rdData[7:0]	<= Latency_r[7:0];
	8'h07:	rdData[7:0]	<= Mask_len_r[7:0];
	8'h08:  rdData[7:0]     <= use_tele_r[7:0];
	8'h09:  rdData[7:0]     <= led_sw_r[7:0];
	8'h0B:  rdData[7:0]     <= trig_delay_r[7:0];
	default:rdData[7:0]	<= 8'b0000_0000;
      endcase // case (REG_ADDR[7:0])
      
      orRv	<= REG_RE;
      orAck	<= REG_WE | REG_RE;
   end
   
   assign	REG_ACK	    = orAck;
   assign	REG_RV	    = orRv;
   assign	REG_RD[7:0] = (orRv ? rdData[7:0] : 8'd0);
   
   //------------------------------------------------------------------------------

endmodule
