`define SYS_VERSION 32'h02_17_0203

module top(
	   //-----------------------------------
	   // SVX4
	   //-----------------------------------
	   //
	   // Bi-Directional LVDS
	   //
	   BUS,//   I/O  : 8bit BUSB
   
	   //
	   // Single ended Signal
	   //
	   PRin,//  OUT  : Priority InB
	   PRout,// IN   : Priority OutB

	   FECLK,// OUT  : FrontEnd ClockB
	   BECLK,// OUT  : BackEnd ClockB

	   OBDV,//  IN  : Odd Byte Data VaridB

	   PARST,//   OUT  : PreAmp Reset
	   PIPERD1,// OUT  : Pipeline Read 1
	   PIPERD2,// OUT  : Pipeline Read 2
	   L1A,//     OUT  : Level one Accept
	   CALSR,//   OUT  : Cal Strobe / Latch Shadow Register

	   FEMODE,//  OUT  : FrontEnd Mode
	   BEMODE,//  OUT  : BackEnd Mode
	   CHMODE,//  OUT  : Mode Change

	   //
	   // Bi Direction Enables
	   //
	   DDATA,//   OUT  : Data Drive Enable
	   RDATA,//   OUT  : Data Receive Enable

	   //---------------------------------------
	   // SEABAS FPGA(USER & SiTCP) CONNECTION
	   //---------------------------------------
	   //
	   // SEABAS IN/OUT Interface
	   //
	   CLK_50M,//       IN 
	   // SW_RSTn,//       IN
	   LED,//           OUT
	   DIP_SW,//        IN
	   NIM_OUT,
	   NIM_IN,

	   //
	   // User I/F
	   //
	   USR_CLK,//       OUT  : Clock
	   USR_ACTIVE,//    IN   : TCP established
	   USR_CLOSE_REQ,// IN   : Request to close a TCP Connection
	   USR_CLOSE_ACK,// OUT  : Acknowledge for USR_CLOSE_REQ

	   //
	   // TCP Tx : TCP Sender
	   //
	   USR_TX_AFULL,//  IN   : Almost full flag of a TCP Tx FIFO
	   USR_TX_WE,//     OUT  : TCP Tx Data write enable
	   USR_TX_WD,//     OUT  : TCP Tx Data[7:0]

	   //
	   // TCP Rx : TCP Receiver
	   //
	   USR_RX_EMPTY,//  IN   : Empty flag of a TCP RX flag
	   //USR_RX_RE,//   OUT  : TCP Rx Data Read enable
	   USR_RX_RV,//     IN   : TCP Rx data valid
	   USR_RX_RD,//     IN   : TCP Rx data[7:0]

	   ////////////////////////////////
	   // Only for simulation !!
	   // ////////////////////////////////
     	   // SIO_ADDR,
    	   // SIO_WD,
    	   // SIO_WE,
	   // end for simulation //////////

	   ADC_PDWN, //     For saving power
	   USR_TESTPOINT,  // For Debug

	   //
	   // Register Access
	   //
	   REG_CLK,//       IN   : Clock
	   REG_ACT,//       IN   : Access Active
	   REG_DI,//        OUT  : Serial Data Output
	   REG_DO//         IN   : Serial Data Input
	   );

   /* ========================================================================== */
   
   ////////////////////////////////
   // Only for simulation !!
   ////////////////////////////////
   // input [31:0] 	SIO_ADDR;
   // input [7:0] 		SIO_WD;
   // input 		SIO_WE;
   // end for simulation //////////
   
   /* ========================================================================== */  

   //-----------------------------------
   // SVX4
   //-----------------------------------
   inout [31:0] 	BUS;

   output [3:0] 	PRin;
   input [3:0] 		PRout;
   
   output [3:0] 	FECLK;
   output [3:0] 	BECLK;
   
   input [3:0] 		OBDV;

   output [3:0] 	PARST;
   output [3:0] 	PIPERD1;
   output [3:0] 	PIPERD2;
   output [3:0] 	L1A;
   output [3:0] 	CALSR;
   
   output [3:0] 	FEMODE;
   output [3:0] 	BEMODE;
   output [3:0] 	CHMODE;
   
   // ---------------------------
   
   reg [31:0] 		bus_r;
   
   reg [3:0] 		prin2_r;//   OUT
   wire [3:0] 		prout_s;//  IN
   reg [3:0] 		feclk2_r;// OUT
   wire 		beclk1_s;// OUT
   wire 		beclk2_s;// OUT
   wire [3:0] 		obdv_s;//   IN
   

   //-----------------------------------
   // SEABAS FPGA(USER & SiTCP) CONNECTION
   //-----------------------------------
   // Bi Direction Enable
   output [3:0] 	DDATA;
   output [3:0] 	RDATA;

   // SEABAS I/O Interface
   input 		CLK_50M;
   // input 		SW_RSTn;
   input [7:0] 		DIP_SW;
   output [7:0] 	LED;
   output [1:0] 	NIM_OUT;
   input [1:0] 		NIM_IN;
   
   wire [7:0] 		DIP_SW_s;
   assign               DIP_SW_s = DIP_SW;

   // User I/F
   output 		USR_CLK;
   input 		USR_ACTIVE;
   input 		USR_CLOSE_REQ;
   output 		USR_CLOSE_ACK;

   // TCP Tx
   input 		USR_TX_AFULL;
   output 		USR_TX_WE;
   output [7:0] 	USR_TX_WD;

   // TCP Rx
   input 		USR_RX_EMPTY;
   //output             USR_RX_RE;
   input 		USR_RX_RV;
   input [7:0] 		USR_RX_RD;

   // Register Access
   input 		REG_CLK;
   input 		REG_ACT;
   output 		REG_DI;
   input 		REG_DO;

   // For SEABAS's ADC power off
   output [1:0]		ADC_PDWN;
   assign               ADC_PDWN[1:0] = 2'b11; // {2{DIP_SW_s[4]}};

   output [3:0] 	USR_TESTPOINT;

   // Assign using channel of D/B (1:use, 0:not use)
   wire [3:0] 		use_tele_s;
   wire [3:0] 		use_tele_via_udp_s;
   assign use_tele_s[3:0]
     = (DIP_SW_s[4] == 1'b1) ? DIP_SW_s[3:0] : use_tele_via_udp_s[3:0];

   wire 		int_trig_mode_s;
   assign int_trig_mode_s = ~DIP_SW_s[7];

   // Trigger Delay
   wire [3:0] 		trigger_delay_width_s;
   
   
   //----------------------------------------
   // Create operation clocks for SVX4
   //----------------------------------------
   // DLL
   wire 		RST_DLL;
   wire  		dcm_rst_s;
   wire 		reset_via_udp_s;
   // assign               RST_DLL = ~SW_RSTn;
   assign               RST_DLL = DIP_SW_s[6] | reset_via_udp_s;
   
   // CLKs created
   wire 		clk_50M_s;
   wire 		clk_50M_b; // From IBUFG output
   // wire 		clk_50M2_b; // From IBUFG output
   
   wire 		clk_40M_s; // main Operational clock
   wire 		clk_40M_90_s; // main Operational clock
   wire 		clk_40M_180_s; // main Operational clock
   wire 		clk_40M_270_s; // main Operational clock
   
   wire 		clk_25M_s; // For USR_TX
   wire 		clk_20M_s; // For USR_TX
   wire 		clk_19M_s; // For USR_TX

   wire 		clk_200M_s; // For Trigger Delay

   wire  		locked_s;
   wire  		locked2_s;
   wire  		locked3_s;
   wire  		locked4_s;

   // =========== System Reset (asynchronous) ============
   wire 		sys_reset_s;
   assign sys_reset_s = ~locked_s;

   // Named Clock
   // wire 	conf_r_clk_s;
   // wire 	ctrl_r_clk_s;
   // wire 	fifo_w_clk_s;
   // wire 	fifo_r_clk_s;
   // assign conf_r_clk_s = clk_40M_s;
   // assign ctrl_r_clk_s = clk_40M_s;
   // assign fifo_w_clk_s = clk_40M_s;
   // assign fifo_r_clk_s = clk_25M_s;
   
   // DCM modules
   clk_dcm clk_dcm(
		   .U1_CLKIN_IN(CLK_50M),
		   .U1_USER_RST_IN(RST_DLL),
		   // .U1_CLKDV_OUT(clk_25M_s),
		   .U1_CLKFX_OUT(clk_40M_s),
                   .U1_CLKIN_IBUFG_OUT(clk_50M_b),
            	   .U1_CLK0_OUT(clk_50M_s),
      		   .U2_CLKFX_OUT(clk_19M_s),
      		   // .U2_CLK0_OUT(),
		   .U2_LOCKED_OUT(locked_s)
		   );

   clk_dcm2 clk_dcm2(
		     .CLKIN_IN(clk_40M_s),
		     .RST_IN(RST_DLL),
		     .CLKFX_OUT(clk_25M_s),// Actually 120MHz
		     .CLKDV_OUT(clk_20M_s),
		     // .CLK0_OUT(),
		     .CLK90_OUT(clk_40M_90_s),
		     .CLK180_OUT(clk_40M_180_s),
		     .CLK270_OUT(clk_40M_270_s),
		     .LOCKED_OUT(locked2_s)
		     );

   clk_dcm3 clk_dcm3(
   		     .CLKIN_IN(clk_40M_s),
   		     .RST_IN(RST_DLL),
   		     .CLKFX_OUT(clk_200M_s), // 200MHz
   		     // .CLK0_OUT(),
   		     .LOCKED_OUT(locked3_s)
   		     );

   dcm_odelay dcm_odelay(
			 .CLKIN_IN(clk_50M_b),
			 .RST_IN(RST_DLL),
			 .CLKFX_OUT(ODELAY_CLK), // 200MHz
			 // .CLKIN_IBUFG_OUT(clk_50M2_s),
			 // .CLK0_OUT(),
			 .LOCKED_OUT(locked4_s)
			 );
   
   // wire 	wait_clksync1_s;
   RST_GEN RST_GEN(
		   .CLK_IN(clk_50M_b),
		   .DCM_LOCKED(locked_s),
		   .DCM_RST(dcm_rst_s)
		   // .WAIT(wait_clksync1_s)
		   );

   assign USR_TESTPOINT = {REG_CLK, REG_ACT, REG_DO, feclk2_r[0]};

   IDELAYCTRL IDELAYCTRL(
			 .REFCLK(ODELAY_CLK),
			 .RST(sys_reset_s),
			 .RDY()
			 );

   
   /* ============================================================= */
   /*                     Data Sending Method                       */
   /* ============================================================= */
   
   //-------------------------------
   // RBCP : register Access
   //-------------------------------   
   wire [31:0] 		SIO_ADDR; 
   wire [7:0] 		SIO_WD;
   wire 		SIO_WE;
   wire 		SIO_RE;
   wire 		SIO_ACK;
   wire 		SIO_RV;
   wire [7:0] 		SIO_RD;
   
   SIO_SLAVE SIO_SLAVE(
		       .RSTn(~sys_reset_s),        // IN : System reset
		       .FILL_ADDR(32'h0),         // IN : Filled address for narow address-width
		       // Serial I/F
		       .SCK(REG_CLK),    	  // IN : Clock
		       .SCS(REG_ACT),    	  // IN : Active
		       .SI(REG_DI),	          // OUT : Data Input
		       .SO(REG_DO),     	  // IN : Data Output
		       // REG I/F
		       .REG_ADDR(SIO_ADDR[31:0]), // OUT : Address[31:0]
		       .REG_WD(SIO_WD[7:0]),	  // OUT : Data[7:0]
		       .REG_WE(SIO_WE),	          // OUT : Write enable
		       .REG_RE(SIO_RE),	          // OUT : Read enable
		       .REG_ACK(SIO_ACK),         // IN : Access acknowledge
		       .REG_RV(SIO_RV),	          // IN : Read valid
		       .REG_RD(SIO_RD[7:0])	  // IN : Read data[7:0]
		       );
   
   ///////////////////////////////
   // UDP register access
   wire [22:0] 		conf_param_s;
   wire 		led_sw_s;
   SIO_REG SIO_REG(
		   .SCK      (REG_CLK),        // in  : Clock
		   .RSTn     (~sys_reset_s),    // in  : System reset
		   // Register I/F
		   .REG_ADDR (SIO_ADDR[31:0]), // in  : Address[31:0]
		   .REG_WD   (SIO_WD[7:0]),    // in  : Data[7:0]
		   .REG_WE   (SIO_WE),         // in  : Write enable
		   .REG_RE   (SIO_RE),         // in  : Read enable
		   .REG_ACK  (SIO_ACK),        // out : Access acknowledge
		   .REG_RV   (SIO_RV),         // out : Read valid
		   .REG_RD   (SIO_RD[7:0]),    // out : Read data[7:0]
		   .CONF_PARAM(conf_param_s[22:0]),
		   .USE_TELE  (use_tele_via_udp_s[3:0]),
		   .TRIG_DELAY(trigger_delay_width_s[3:0]),
		   .LED_SW    (led_sw_s),
		   .SYS_VER   (`SYS_VERSION)    // in  : System version info
		   );
   
   
   //-----------------------------------------
   // State Flags and COMMANDs declaration
   //-----------------------------------------
   
   ///////////////////////////////////////////////
   // Memory switcher
   //
   wire 		command_valid_s;
   wire [3:0] 		conf_we_s;
   wire 		ctrl_we_s;
   // wire 		reset_via_udp_s;

   assign reset_via_udp_s = ((SIO_WE == 1'b1)
			     && (SIO_ADDR[31:0] == 32'h10a)) ? 1'b1 : 1'b0;

   assign command_valid_s = ~(|SIO_ADDR[31:12]) & SIO_WE;
   
   assign conf_we_s[0] = ~(|SIO_ADDR[31:16]) & (~SIO_ADDR[15] & ~SIO_ADDR[14] & ~SIO_ADDR[13] & SIO_ADDR[12]) & SIO_WE;
   assign conf_we_s[1] = ~(|SIO_ADDR[31:16]) & (~SIO_ADDR[15] & ~SIO_ADDR[14] & SIO_ADDR[13] & ~SIO_ADDR[12]) & SIO_WE;
   assign conf_we_s[2] = ~(|SIO_ADDR[31:16]) & (~SIO_ADDR[15] & SIO_ADDR[14] & ~SIO_ADDR[13] & ~SIO_ADDR[12]) & SIO_WE;
   assign conf_we_s[3] = ~(|SIO_ADDR[31:16]) & (SIO_ADDR[15] & ~SIO_ADDR[14] & ~SIO_ADDR[13] & ~SIO_ADDR[12]) & SIO_WE;

   assign ctrl_we_s = ~(|SIO_ADDR[31:17]) & SIO_ADDR[16] & SIO_WE;
   
   ///////////////////////////////////////////////
   // Register for Commands
   //
   parameter [11:0] COMMAND = 12'ha5a;

   parameter [7:0]  COM_ACTIVE = 8'hff;
   parameter [7:0]  COM_DEACTIVE = 8'h00;

   parameter [7:0]  COM_CONTROL = 8'h01;
   parameter [7:0]  COM_CONTROL_STOP = 8'h03;
   
   parameter [7:0]  COM_CONFIGURE = 8'h05;
   parameter [7:0]  COM_START = 8'h55;
   parameter [7:0]  COM_PAUSE = 8'h5a;
   parameter [7:0]  COM_RESUME = 8'ha5;
   parameter [7:0]  COM_STOP = 8'haa;
   parameter [7:0]  COM_UNCONFIGURE = 8'ha0;
   
   reg 			conf_start_r;	//
   reg 			conf_reading_r;	//
   reg 			unconfigure_r;	// 
   
   reg 			start_daq_r;	//
   reg 			pause_daq_r;	// 
   reg 			stop_daq_r;	//
   
   reg 			DAQ_Loaded_r;	// DAQLifeCycleState LOADED
   reg 			DAQ_Ready_r;	// DAQLifeCycleState CONFIGURED
   reg 			DAQ_Running_r;	// DAQLifeCycleState RUNNING
   reg 			DAQ_Paused_r;	// DAQLifeCycleState PAUSED

   ////////////////////////////////////
   // Main Active Flag
   reg 			main_active_r = 1'b0;
   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 main_active_r <= 1'b0;
      end else begin
	 if(main_active_r == 1'b0)begin
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND
	       && SIO_WD[7:0] == COM_ACTIVE)begin
	       main_active_r <= 1'b1;
	    end
	 end else if(main_active_r == 1'b1)begin
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND
	       && SIO_WD[7:0] == COM_DEACTIVE)begin
	       main_active_r <= 1'b0;
	    end
	 end // if (main_active_r == 1'b1)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)
   
   wire 	buf_main_active_40M_s;
   wire 	buf_main_active_20M_s; // To be removed
   double_ff dff_mainact0(
			  .idata(main_active_r),
			  .oclk(clk_40M_s),
			  .odata(buf_main_active_40M_s)
			  );
   // To be removed
   double_ff dff_mainact1(
			  .idata(main_active_r),
			  .oclk(clk_20M_s),
			  .odata(buf_main_active_20M_s)
			  );
   
   
   //-------------------------------
   // SVX mode operator
   //-------------------------------
   reg          bemode_in_r;
   reg 		femode_in_r;

   wire 	bemode_out_s;
   wire 	femode_out_s;
   wire 	chmode_out_s;

   wire 	init_mode_s;
   wire 	acq_mode_s;
   wire 	digit_mode_s;
   wire 	read_mode_s;
   wire 	rst_mode_s;

   wire 	running_flag_s;
   wire 	mode_error_s;

   mode_changer mode_changer(
			     .OPERATE_CLK_IN(clk_40M_s),
			     .ACTIVE_IN(buf_main_active_40M_s),
			     .RESET_IN(sys_reset_s),
			     .BEMODE_IN(bemode_in_r),
			     .FEMODE_IN(femode_in_r),
			     .RUN_FLAG_IN(running_flag_s),
			     .BEMODE_OUT(bemode_out_s),
			     .FEMODE_OUT(femode_out_s),
			     .CHMODE_OUT(chmode_out_s),
			     .INIT_FLAG(init_mode_s),
			     .ACQ_FLAG(acq_mode_s),
			     .DIGIT_FLAG(digit_mode_s),
			     .READ_FLAG(read_mode_s),
			     .RST_FLAG(rst_mode_s),
			     .ERROR_OUT(mode_error_s)
			     );
   
   
   //-------------------------------
   // SVX configuration
   //-------------------------------
   wire [3:0] 	conf_RAM_ready_s;
   wire [3:0] 	configured_s;
   wire [3:0] 	conf_calsr_s;
   wire [3:0] 	conf_prin_s;
   wire [3:0] 	conf_feclk_s;

   wire 	all_configured_s;
   assign all_configured_s
     = (configured_s[3] | ~use_tele_s[3]) & (configured_s[2] | ~use_tele_s[2])
       & (configured_s[1] | ~use_tele_s[1]) & (configured_s[0] | ~use_tele_s[0]);
   
   conf_maker conf_maker0(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(buf_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[0]),
			  .PROUT_IN(prout_s[0]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[0]),
			  .CONFIGURED_OUT(configured_s[0]),
			  .CALSR_OUT(conf_calsr_s[0]),
			  .PRIN_OUT(conf_prin_s[0]),
			  .FECLK_OUT(conf_feclk_s[0])
			  );

   conf_maker conf_maker1(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(buf_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[1]),
			  .PROUT_IN(prout_s[1]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[1]),
			  .CONFIGURED_OUT(configured_s[1]),
			  .CALSR_OUT(conf_calsr_s[1]),
			  .PRIN_OUT(conf_prin_s[1]),
			  .FECLK_OUT(conf_feclk_s[1])
			  );

   conf_maker conf_maker2(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(buf_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[2]),
			  .PROUT_IN(prout_s[2]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[2]),
			  .CONFIGURED_OUT(configured_s[2]),
			  .CALSR_OUT(conf_calsr_s[2]),
			  .PRIN_OUT(conf_prin_s[2]),
			  .FECLK_OUT(conf_feclk_s[2])
			  );

   conf_maker conf_maker3(
			  .OPERATE_CLK_IN(clk_20M_s),
			  .REG_CLK_IN(REG_CLK),
			  .ACTIVE_IN(buf_main_active_20M_s),
			  .CONF_START_IN(conf_start_r),
			  .RESET_IN(sys_reset_s | unconfigure_r),
			  .SIO_ADDR_IN(SIO_ADDR[11:0]),
			  .SIO_WD_IN(SIO_WD[7:0]),
			  .SIO_WE_IN(conf_we_s[3]),
			  .PROUT_IN(prout_s[3]),
			  .RAM_LOADED_OUT(conf_RAM_ready_s[3]),
			  .CONFIGURED_OUT(configured_s[3]),
			  .CALSR_OUT(conf_calsr_s[3]),
			  .PRIN_OUT(conf_prin_s[3]),
			  .FECLK_OUT(conf_feclk_s[3])
			  );
   
   //////////////////////////////////////////////
   // Send PRout & FECLK to PC
   reg [7:0] 	conf_fifo_in_r;
   reg 		conf_fifo_re_r;
   
   wire [7:0] 	conf_fifo_out_s;
   wire 	conf_fifo_full_s;
   wire 	conf_fifo_empty_s;
   wire 	conf_fifo_valid_s;

   wire 	buf_init_mode_s;
   double_ff dff_init(
		    .idata(init_mode_s),
		    .oclk(clk_20M_s),
		    .odata(buf_init_mode_s)
		    );
   
   fifo_conf fifo_conf(
		       .rst(sys_reset_s
			    | ~USR_ACTIVE
			    | unconfigure_r),        // IN   : Reset FIFO
		       .wr_clk(clk_20M_s),           // IN   : Clock for write data 
		       .rd_clk(clk_25M_s),           // IN   : Clock for readout data
		       .din(conf_fifo_in_r[7:0]),    // IN   : Input data
		       .wr_en(buf_init_mode_s
			      & ~all_configured_s),  // IN   : Input write enable
		       .rd_en(conf_fifo_re_r),       // IN   : Output read enable
		       .dout(conf_fifo_out_s[7:0]),  // OUT  : Output data
		       .full(conf_fifo_full_s),      // OUT  : FIFO full
		       .empty(conf_fifo_empty_s),    // OUT  : FIFO empty
		       .valid(conf_fifo_valid_s)     // OUT  : Data Valid
		       );

   wire 	buf_running_flag_s;
   double_ff dff_run(
		      .idata(running_flag_s),
		      .oclk(clk_25M_s),
		      .odata(buf_running_flag_s)
		      );

   always @(posedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_fifo_re_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    conf_fifo_re_r <= 1'b0;
	 end else begin
	    
	    if(USR_TX_AFULL == 1'b0 && buf_running_flag_s == 1'b0)begin
	       if(conf_fifo_empty_s == 1'b0)begin
		  conf_fifo_re_r <= 1'b1;
	       end
	    end else begin
	       conf_fifo_re_r <= 1'b0;
	    end
	    
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_25M_s or posedge sys_reset_s)

   always @(posedge clk_20M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_fifo_in_r[7:0] <= 8'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    conf_fifo_in_r[7:0] <= 8'b0;
	 end else begin
	    conf_fifo_in_r[7:0] <= {conf_feclk_s[3], prout_s[3],
				    conf_feclk_s[2], prout_s[2],
				    conf_feclk_s[1], prout_s[1],
				    conf_feclk_s[0], prout_s[0]};
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_20M_s or posedge sys_reset_s)
   
   
   //--------------------------
   // Trigger
   //--------------------------
   
   ///////////////////////
   // Delay 
   reg [1:0]  trigger_predelay_shifter_r;
   reg [31:0] trigger_delay_shifter_r;

   function delay_maker_f;
      input [3:0] delay_width;
      case(delay_width)
	4'd0: delay_maker_f = trigger_delay_shifter_r[0];
	4'd1: delay_maker_f = trigger_delay_shifter_r[2];
	4'd2: delay_maker_f = trigger_delay_shifter_r[4];
	4'd3: delay_maker_f = trigger_delay_shifter_r[6];
	4'd4: delay_maker_f = trigger_delay_shifter_r[8];
	4'd5: delay_maker_f = trigger_delay_shifter_r[10];
	4'd6: delay_maker_f = trigger_delay_shifter_r[12];
	4'd7: delay_maker_f = trigger_delay_shifter_r[14];
	4'd8: delay_maker_f = trigger_delay_shifter_r[16];
	4'd9: delay_maker_f = trigger_delay_shifter_r[18];
	4'd10: delay_maker_f = trigger_delay_shifter_r[20];
	4'd11: delay_maker_f = trigger_delay_shifter_r[22];
	4'd12: delay_maker_f = trigger_delay_shifter_r[24];
	4'd13: delay_maker_f = trigger_delay_shifter_r[26];
	4'd14: delay_maker_f = trigger_delay_shifter_r[28];
	4'd15: delay_maker_f = trigger_delay_shifter_r[30];
	default:  delay_maker_f = trigger_delay_shifter_r[0];
      endcase // case (delay_width)
   endfunction // case

   always @(posedge clk_200M_s)begin
      if(sys_reset_s == 1'b1)begin
	 trigger_predelay_shifter_r[1:0] <= 2'b0;
	 trigger_delay_shifter_r[31:0] <= 16'b0;
      end else begin
	 trigger_predelay_shifter_r[1:0] <= {trigger_predelay_shifter_r[0], ~NIM_IN[0]};
	 trigger_delay_shifter_r[31:0] <= {trigger_delay_shifter_r[30:0], trigger_predelay_shifter_r[1]};
      end
   end

   wire nim_trigger_s;
   assign nim_trigger_s = delay_maker_f(trigger_delay_width_s[3:0]);
   

   ////////////////////////////
   // Trigger Module
   wire        triged_s;
   wire        pulse_trigger_s;
   reg 	       trigger_rcved_r;
   
   wire        trig_waiting_s;
   wire        busy_out_s;

   wire        ts_clk_s;
   assign ts_clk_s = ~NIM_IN[1];

   wire        header_we_s;
   reg 	       evheader_writing_r;
   wire [7:0]  event_header_s;
   reg [7:0]   event_header_r;

   wire [7:0]  status_s;

   trigger trigger(
		   .TRIGGER_CLK(clk_40M_s),
		   .OPERATE_CLK(clk_40M_s),
		   .HEADER_CLK(clk_40M_s),
		   .TS_CLK(ts_clk_s),
      
		   .ACTIVE_IN(USR_ACTIVE),
		   .BUSY_IN(busy_out_s),
		   // .BUSY_IN(triged_s),
      
		   .START_IN(start_daq_r),
		   .PAUSE_IN(pause_daq_r),
		   .STOP_IN(stop_daq_r),
		   .RESET_IN(sys_reset_s),
		   .INT_TRIG_IN(int_trig_mode_s),
      
		   .NIM_TRIGGER(~nim_trigger_s),
		   .TRIGGER_RCVED_IN(trigger_rcved_r),
		   
		   .TRIGGER_OUT(pulse_trigger_s),
		   .TRIGGERED_FLAG(triged_s),
		   .RUNNING_FLAG(running_flag_s),
      
		   .STATUS_IN(status_s[7:0]),
		   .HEADER_OUT(event_header_s[7:0]),
		   .HEADER_WE(header_we_s)
		   );

   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 trigger_rcved_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    trigger_rcved_r <= 1'b0;
	 end else begin
	    
	    if(triged_s == 1'b1/*Triggerd flag*/)begin
	       if(digit_mode_s == 1'b1/*Exit Trigger Waiting Loop*/)begin
	    	  trigger_rcved_r <= 1'b1;
	       end
	    end else if(read_mode_s == 1'b1)begin
	       trigger_rcved_r <= 1'b0;
	    end

	 end // else: !if(USR_ACTIVE == 1'b1)
      end // if (sys_reset_s == 1'b0)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)

   always@(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 event_header_r[7:0] <= 8'b0;
	 evheader_writing_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    event_header_r[7:0] <= 8'b0;
	    evheader_writing_r <= 1'b0;
	 end else begin
	    event_header_r[7:0] <= event_header_s[7:0];
	    evheader_writing_r <= header_we_s;
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always@ (posedge clk_40M_s or posedge sys_reset_s)


   //////////////////////////////////////////
   // Pedestal Stabilizer
   wire 	  buf_all_configured_s;
   double_ff dff_confed(
		    .idata(all_configured_s),
		    .oclk(clk_40M_s),
		    .odata(buf_all_configured_s)
		    );

   wire 	  buf_reset_sw_s;
   double_ff dff_reset_sw(
			  .idata(DIP_SW_s[5]),
			  .oclk(clk_40M_s),
			  .odata(buf_reset_sw_s)
			  );
   
   reg reset_trigger_r;
   reg reset_busy_r;
   reg reset_read_m_end_r;
   reg [23:0] reset_counter_r;
   parameter [23:0] RESET_COUNT = 24'd353982; // ~113Hz
   always@(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 reset_trigger_r <= 1'b0;
	 reset_busy_r <= 1'b0;
	 reset_read_m_end_r <= 1'b0;
	 reset_counter_r[23:0] <= 24'd0;
      end else begin
	 if(USR_ACTIVE == 1'b0 | buf_reset_sw_s == 1'b0)begin
	    reset_trigger_r <= 1'b0;
	    reset_busy_r <= 1'b0;
	    reset_read_m_end_r <= 1'b0;
	    reset_counter_r[23:0] <= 24'd0;
	 end else begin
	    
	    if(triged_s == 1'b1 || reset_counter_r[23:0] == RESET_COUNT || 
	       buf_all_configured_s == 1'b0)begin
	       reset_counter_r[23:0] <= 24'd0;
	    end else begin
	       reset_counter_r[23:0] <= reset_counter_r[23:0] + 24'd1;
	    end
	    
	    if(reset_counter_r[23:0] == 24'd0)begin
	       reset_trigger_r <= 1'b0;
	       reset_busy_r <= 1'b0;
	       reset_read_m_end_r <= 1'b0;
	    end else if(reset_counter_r[23:0] == (RESET_COUNT - 24'd2000))begin
	       reset_busy_r <= 1'b1;
	    end else if(reset_counter_r[23:0] == (RESET_COUNT - 24'd2000 + 24'd1))begin
	       reset_trigger_r <= 1'b1;
	    end else if(reset_counter_r[23:0] == (RESET_COUNT - 24'd2000 + 24'd3))begin
	       reset_trigger_r <= 1'b0;
	    end else if(reset_counter_r[23:0] == (RESET_COUNT - 24'd5))begin
	       reset_read_m_end_r <= 1'b1;
	    end

	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always@ (posedge clk_40M_s or posedge sys_reset_s)
   
   
   //--------------------------
   // SVX control signals
   //--------------------------
   wire 	  ctrl_RAM_ready_s;
   wire [15:0] 	  ctrl_sig_s;
   
   wire 	  fifo_all_r_ready_s;
   wire 	  buf_fifo_all_r_ready_s;
   double_ff dff_fifo_all_r_ready(
				  .idata(fifo_all_r_ready_s),
				  .oclk(clk_40M_s),
				  .odata(buf_fifo_all_r_ready_s)
				  );

   // wire 	  mux_clock_s;
   // BUFGMUX_CTRL BUFGMUX_inst(.O(mux_clock_s), .S(buf_init_mode_s),
   // 			     .I0(clk_40M_s), .I1(clk_20M_s));
   
   wire [1:0] 	  feclk_cnt_fine_s;
   multisample_TDC feclk_cnt (
			      .CLK_0_IN(clk_40M_s), 
			      .CLK_90_IN(clk_40M_90_s), 
			      .CLK_180_IN(clk_40M_180_s),  
			      .CLK_270_IN(clk_40M_270_s),  
			      .START_IN(nim_trigger_s),
			      .DATA_OUT(feclk_cnt_fine_s[1:0])
			      );
   
   wire [3:0] 	  feclk_cnt_s;
   wire 	  trigger_in_s;
   wire 	  read_m_end_s;
   wire           parst_busy_s;	// For SOFIST Beamtest
   assign status_s[7:0] = {3'b0, feclk_cnt_s[2:0], feclk_cnt_fine_s[1:0]};
   assign trigger_in_s = reset_trigger_r | triged_s;
   assign read_m_end_s = reset_read_m_end_r | buf_fifo_all_r_ready_s;
   
   ctrl_sig_maker ctrl_sig_maker(
				 .OPERATE_CLK_IN(clk_40M_s),
				 .REG_CLK_IN(REG_CLK),
				 .ACTIVE_IN(buf_main_active_40M_s),
				 .RESET_IN(sys_reset_s | unconfigure_r),
				 .SIO_ADDR_IN(SIO_ADDR[13:0]),
				 .SIO_WD_IN(SIO_WD[7:0]),
				 .SIO_WE_IN(ctrl_we_s),
				 .TRIGGER_IN(trigger_in_s),
				 .READ_START_IN(buf_all_configured_s),
				 .READ_M_END_IN(read_m_end_s),
				 .CONF_PARAM_IN(conf_param_s[22:0]),
				 .RAM_LOADED_OUT(ctrl_RAM_ready_s),
				 .TRIG_WAIT_OUT(trig_waiting_s),
				 .PARST_BUSY_OUT(parst_busy_s),
				 .FECLK_CNT_OUT(feclk_cnt_s[3:0]),
				 .CTRL_SIG_OUT(ctrl_sig_s[15:0])
				 );

   // 141208 Yajima : This section added
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 bemode_in_r <= 1'b1;
	 femode_in_r <= 1'b0;
      end else begin
	 if(buf_main_active_40M_s == 1'b0)begin
	    bemode_in_r <= 1'b1;
	    femode_in_r <= 1'b0;
	 end else begin

	    if(buf_all_configured_s == 1'b1)begin
	       if(ctrl_sig_s[15]/*CHMODE*/ == 1'b1)begin
		  bemode_in_r <= ctrl_sig_s[13];
		  femode_in_r <= ctrl_sig_s[14];
	       end
	    end else if(conf_start_r == 1'b1)begin
	       bemode_in_r <= 1'b0;
	       femode_in_r <= 1'b0;
	    end

	 end // else: !if(buf_main_active_40M_s == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)

   reg [7:0] ctrl_sig_inv_r;
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 ctrl_sig_inv_r[7:0] <= 8'b0011_0011;
      end else begin
	 if(buf_main_active_40M_s == 1'b0)begin
	    ctrl_sig_inv_r[7:0] <= 8'b0011_0011;
	 end else begin
	    ctrl_sig_inv_r[7:0] <= {ctrl_sig_s[0], ctrl_sig_s[1],
				    ctrl_sig_s[2], ctrl_sig_s[3],
				    ctrl_sig_s[4], ctrl_sig_s[5],
				    ctrl_sig_s[6], ctrl_sig_s[7]};
	 end // else: !if(buf_main_active_40M_s == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)

   wire       buf0_conf_calsr_s;
   wire       buf1_conf_calsr_s;
   wire       buf2_conf_calsr_s;
   wire       buf3_conf_calsr_s;
   double_ff dff_calsr0(.idata(conf_calsr_s[0]),
		    .oclk(clk_40M_s), .odata(buf0_conf_calsr_s)
		    );
   double_ff dff_calsr1(.idata(conf_calsr_s[1]),
			.oclk(clk_40M_s), .odata(buf1_conf_calsr_s)
			);
   double_ff dff_calsr2(.idata(conf_calsr_s[2]),
			.oclk(clk_40M_s), .odata(buf2_conf_calsr_s)
			);
   double_ff dff_calsr3(.idata(conf_calsr_s[3]),
			.oclk(clk_40M_s), .odata(buf3_conf_calsr_s)
			);
   
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 bus_r[31:0] <= 32'b0011_0011_0011_0011_0011_0011_0011_0011;
      end else begin
	 if(buf_main_active_40M_s == 1'b0)begin
	    bus_r[31:0] <= 32'b0011_0011_0011_0011_0011_0011_0011_0011;	    
	 end else begin
	    bus_r[31:0] <= (init_mode_s == 1'b1 && buf_all_configured_s == 1'b0) 
	      ? {buf3_conf_calsr_s, 7'b011_0011,
		 buf2_conf_calsr_s, 7'b011_0011,
		 buf1_conf_calsr_s, 7'b011_0011,
		 buf0_conf_calsr_s, 7'b011_0011} : { 4{ctrl_sig_inv_r[7:0]} };
	 end // else: !if(buf_main_active_40M_s == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)

   reg 	    beclk_r;
   reg      feclk_r;
   reg 	    clk_sel_r;
   reg 	    prin_r;
   
   always @(posedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 beclk_r <= 1'b0;
	 feclk_r <= 1'b0;
	 clk_sel_r <= 1'b0;
	 prin_r <= 1'b0;
      end else begin
	 if(buf_main_active_40M_s == 1'b0)begin
	    beclk_r <= 1'b0;
	    feclk_r <= 1'b0;
	    clk_sel_r <= 1'b0;
	    prin_r <= 1'b0;
	 end else begin
	    beclk_r <= ctrl_sig_s[9];
	    feclk_r <= ctrl_sig_s[10];
	    clk_sel_r <= ctrl_sig_s[11];
	    prin_r <= ctrl_sig_s[12];
	 end // else: !if(buf_main_active_40M_s == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_40M_s or posedge sys_reset_s)


   always @(posedge clk_40M_s)begin
      feclk2_r[3:0] = (init_mode_s == 1'b1 || conf_start_r == 1'b1) ? ~conf_feclk_s[3:0] : { 4{~feclk_r} };
      prin2_r[3:0] = (init_mode_s == 1'b1) ? ~conf_prin_s[3:0] : { 4{~prin_r} };
   end
   
   /* ============================================================= */
   /*                     Data Receiving Method                     */
   /* ============================================================= */
   
   //---------------------------------
   // Fifo
   //---------------------------------
   wire [31:0] fifo_out_s;
   wire [3:0]  fifo_full_s;
   wire [3:0]  fifo_empty_s;
   wire [3:0]  fifo_dvalid_s;
   wire [3:0]  fifo_r_ready_s;
   reg [3:0]   fifo_re_r;

   wire        buf_evheader_writing_s;
   double_ff dff_evhead_w(
			  .idata(evheader_writing_r),
			  .oclk(clk_25M_s),
			  .odata(buf_evheader_writing_s)
			  );

   reg buf_evheader_writing_r;
   always @(negedge clk_40M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 buf_evheader_writing_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    buf_evheader_writing_r <= 1'b0;
	 end else begin
	    buf_evheader_writing_r <= evheader_writing_r;
	 end
      end
   end // always @ (negedge clk_40M_s or posedge sys_reset_s)

   wire [31:0] bus_data_s;

   wire [7:0]  fifo_mod0_in_s;
   wire        read_mode_fifo_s;
   assign fifo_mod0_in_s[7:0] = (read_mode_s == 1'b1) ?
				bus_data_s[7:0] : event_header_r[7:0];
   assign read_mode_fifo_s = (read_mode_s & ~reset_busy_r);
   
   fifo_mod fifo_mod0(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(fifo_mod0_in_s[7:0]),
		      .READ_MODE_IN(read_mode_fifo_s),
		      .FIFO_RE_IN(fifo_re_r[0]),
		      .OBDV_IN(obdv_s[0]),
		      .FIFO_FORCE_WE(buf_evheader_writing_r),
		      .OUTPUT_DATA(fifo_out_s[7:0]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[0]),
		      .FIFO_FULL_OUT(fifo_full_s[0]),
		      .DATA_VALID_OUT(fifo_dvalid_s[0]),
		      .FIFO_READ_READY(fifo_r_ready_s[0])
		      );

   fifo_mod fifo_mod1(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(bus_data_s[15:8]),
		      .READ_MODE_IN(read_mode_fifo_s),
		      .FIFO_RE_IN(fifo_re_r[1]),
		      .OBDV_IN(obdv_s[1]),
		      .FIFO_FORCE_WE(1'b0),
		      .OUTPUT_DATA(fifo_out_s[15:8]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[1]),
		      .FIFO_FULL_OUT(fifo_full_s[1]),
		      .DATA_VALID_OUT(fifo_dvalid_s[1]),
		      .FIFO_READ_READY(fifo_r_ready_s[1])
		      );
   
   fifo_mod fifo_mod2(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(bus_data_s[23:16]),
		      .READ_MODE_IN(read_mode_fifo_s), 
		      .FIFO_RE_IN(fifo_re_r[2]),
		      .OBDV_IN(obdv_s[2]),
		      .FIFO_FORCE_WE(1'b0),
		      .OUTPUT_DATA(fifo_out_s[23:16]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[2]),
		      .FIFO_FULL_OUT(fifo_full_s[2]),
		      .DATA_VALID_OUT(fifo_dvalid_s[2]),
		      .FIFO_READ_READY(fifo_r_ready_s[2])
		      );
   
   fifo_mod fifo_mod3(
		      .OPERATE_CLK_IN(clk_40M_s),
		      .READOUT_CLK_IN(clk_25M_s),
      		      .ACTIVE_IN(USR_ACTIVE),
		      .RESET_IN(sys_reset_s),
		      .INPUT_DATA(bus_data_s[31:24]),
		      .READ_MODE_IN(read_mode_fifo_s), 
		      .FIFO_RE_IN(fifo_re_r[3]),
		      .OBDV_IN(obdv_s[3]),
		      .FIFO_FORCE_WE(1'b0),
		      .OUTPUT_DATA(fifo_out_s[31:24]),
		      .FIFO_EMPTY_OUT(fifo_empty_s[3]),
		      .FIFO_FULL_OUT(fifo_full_s[3]),
		      .DATA_VALID_OUT(fifo_dvalid_s[3]),
		      .FIFO_READ_READY(fifo_r_ready_s[3])
		      );
   
   wire        fifo_all_read_s;
   assign fifo_all_r_ready_s
     = (fifo_r_ready_s[3] | ~use_tele_s[3]) & (fifo_r_ready_s[2] | ~use_tele_s[2])
       & (fifo_r_ready_s[1] | ~use_tele_s[1]) & (fifo_r_ready_s[0] | ~use_tele_s[0]);
   assign fifo_all_read_s
     = ~((fifo_r_ready_s[3] & use_tele_s[3]) | (fifo_r_ready_s[2] & use_tele_s[2])
	 | (fifo_r_ready_s[1] & use_tele_s[1]) | (fifo_r_ready_s[0] & use_tele_s[0]));
   
   reg 	       data_reading_r;
   always @(posedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 fifo_re_r[3:0] <= 4'b0000;
	 data_reading_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    fifo_re_r[3:0] <= 4'b0000;
	    data_reading_r <= 1'b0;
	 end else begin
	    
	    if(USR_TX_AFULL == 1'b0 && buf_evheader_writing_s == 1'b0)begin
	       if(fifo_empty_s[0] == 1'b0
		  && use_tele_s[0] == 1'b1 && fifo_all_r_ready_s == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0001;
		  data_reading_r <= 1'b1;
	       end else if(fifo_empty_s[1] == 1'b0
			   && data_reading_r == 1'b1 && use_tele_s[1] == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0010;
	       end else if(fifo_empty_s[2] == 1'b0
			   && data_reading_r == 1'b1 && use_tele_s[2] == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0100;
	       end else if(fifo_empty_s[3] == 1'b0
			   && data_reading_r == 1'b1 && use_tele_s[3] == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b1000;
	       end else if(fifo_all_read_s == 1'b1)begin
		  fifo_re_r[3:0] <= 4'b0000;
		  data_reading_r <= 1'b0;
	       end else begin
		  fifo_re_r[3:0] <= 4'b0000;
	       end
	    end else begin
	       fifo_re_r[3:0] <= 4'b0000;
	    end // else: !if(USR_TX_AFULL == 1'b0 && evheader_writing_r == 1'b0)
	    
	 end // else: !if(USR_ACTIVE == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge clk_25M_s or posedge sys_reset_s)


   //---------------------------------
   // Signal assignments
   //---------------------------------
   ///////////////////////////////////////////
   // BUSY
   wire    all_fifo_empty_s;
   
   assign all_fifo_empty_s
     = (fifo_empty_s[3] | ~use_tele_s[3]) & (fifo_empty_s[2] | ~use_tele_s[2])
       & (fifo_empty_s[1] | ~use_tele_s[1]) & (fifo_empty_s[0] | ~use_tele_s[0]);
   assign busy_out_s =( ~(trig_waiting_s & ~triged_s &
			  ~reset_busy_r & running_flag_s & all_fifo_empty_s) )
     & buf_main_active_40M_s;
   

   ///////////////////////////////////////////
   // Signals
   // assign BEMODE[3:0] = { 4{bemode_out_s} };
   // assign FEMODE[3:0] = { 4{femode_out_s} };
   // assign CHMODE[3:0] = { 4{chmode_out_s} };
   
   // assign beclk_s = (clk_sel_r == 1'b1) ? clk_40M_s : beclk_r;

   assign DDATA[3:0] = (read_mode_s == 1'b1) ? 4'b1111 : 4'b0000;
   assign RDATA[3:0] = (read_mode_s == 1'b0) ? 4'b1111 : 4'b0000;
   
   ///////////////////////////////////////////
   // BUS Lines
   wire [31:0] bus_s;
   assign bus_s[31:0] = ~bus_r[31:0];

   assign PARST[3:0] =   (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[28], ~bus_s[20], ~bus_s[12], ~bus_s[4]};
   assign PIPERD1[3:0] = (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[30], ~bus_s[22], ~bus_s[14], ~bus_s[6]};
   assign PIPERD2[3:0] = (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[26], ~bus_s[18], ~bus_s[10], ~bus_s[2]};
   assign L1A[3:0] =     (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[29], ~bus_s[21], ~bus_s[13], ~bus_s[5]};
   assign CALSR[3:0] =   (read_mode_s == 1'b1) ? 4'b0 :
			 {~bus_s[31], ~bus_s[23], ~bus_s[15], ~bus_s[7]};

   assign LED[7:0] = (led_sw_s == 1'b1) ? {init_mode_s, acq_mode_s, digit_mode_s, read_mode_s
					   , use_tele_s[3:0]} : 8'b0000_0000;
   
   
   //////////////////////////////////////////
   // NIM Output
   reg [15:0]  test_cnt_r;
   always @(posedge clk_40M_s)begin
      if(sys_reset_s == 1'b1)begin
	 test_cnt_r[15:0] <= 16'b0;
      end else begin
	 if(feclk2_r[0] == 1'b0)begin
	    test_cnt_r[15:0] <= test_cnt_r[15:0] + 16'd1;
	 end
      end
   end
      
   assign NIM_OUT[0] = ~busy_out_s;
   assign NIM_OUT[1] = ~parst_busy_s;
   // assign NIM_OUT[1] = (~(|test_cnt_r[12:0]) & ~busy_out_s) ? feclk2_r[0] : 1'b1; // For Debug
      
   //////////////////////////////////////////
   // define BUS I/O selector
   wire        io_state_sel_s;
   assign io_state_sel_s = read_mode_s;
   
   
   //----------------------------------------------------
   //  TC sender : Send data to PC with SiTCP method
   //----------------------------------------------------
   
   reg [7:0]   TX_WD_r;

   function [7:0] TX_Data_sw_f;
      input [5:0] TX_Data_sw;
      case(TX_Data_sw)
	6'b100000: TX_Data_sw_f = conf_fifo_out_s[7:0];
	// 6'b010000: TX_Data_sw_f = event_header_r[7:0];
	6'b001000: TX_Data_sw_f = fifo_out_s[31:24];
	6'b000100: TX_Data_sw_f = fifo_out_s[23:16];
	6'b000010: TX_Data_sw_f = fifo_out_s[15:8];
	6'b000001: TX_Data_sw_f = fifo_out_s[7:0];
	default: TX_Data_sw_f = 8'd0;
      endcase // case (TX_Data_sw)
   endfunction // case

   always @(negedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 TX_WD_r[7:0] <= 8'd0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    TX_WD_r[7:0] <= 8'd0;
	 end else begin
	    TX_WD_r <= TX_Data_sw_f({conf_fifo_valid_s, 1'b0, fifo_dvalid_s[3:0]});
	 end
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (negedge clk_25M_s or posedge sys_reset_s)

   reg buf_conf_fifo_valid_r;
   // reg buf_evheader_writing_r;
   reg buf_fifo_dvalid_r;
   
   always @(negedge clk_25M_s or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 buf_conf_fifo_valid_r <= 1'b0;
	 // buf_evheader_writing_r <= 1'b0;
	 buf_fifo_dvalid_r <= 1'b0;
      end else begin
	 if(USR_ACTIVE == 1'b0)begin
	    buf_conf_fifo_valid_r <= 1'b0;
	    // buf_evheader_writing_r <= 1'b0;
	    buf_fifo_dvalid_r <= 1'b0;
	 end else begin
	    buf_conf_fifo_valid_r <= conf_fifo_valid_s;
	    // buf_evheader_writing_r <= evheader_writing_r;
	    buf_fifo_dvalid_r <= |fifo_dvalid_s;
	 end
      end
   end
   
   assign USR_CLK = clk_25M_s;
   assign USR_TX_WD[7:0] = TX_WD_r[7:0];
   assign USR_TX_WE = buf_fifo_dvalid_r |
		      buf_conf_fifo_valid_r;
   // | buf_evheader_writing_r;

   assign USR_CLOSE_ACK = USR_CLOSE_REQ;


   //---------------------------------
   // State Flags and COMMANDs
   //---------------------------------
   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_reading_r <= 1'b0;	 
      end else begin
	 if(main_active_r == 1'b0)begin
	    conf_reading_r <= 1'b0;
	 end else begin
	    
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND)begin
	       if(SIO_WD[7:0] == COM_CONTROL_STOP)begin
		  conf_reading_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_CONTROL
			   || SIO_WD[7:0] == COM_UNCONFIGURE)begin
		  conf_reading_r <= 1'b1;
	       end
	    end
	    
	 end // else: !if(main_active_r == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)

   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 conf_start_r <= 1'b0;
      end else begin
	 if(main_active_r == 1'b0)begin
	    conf_start_r <= 1'b0;
	 end else begin
	    
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND
	       && SIO_WD[7:0] == COM_CONFIGURE)begin
	       conf_start_r <= 1'b1;
	    end else begin
	       conf_start_r <= 1'b0;
	    end
	    
	 end // else: !if(main_active_r == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)

   always @(posedge REG_CLK or posedge sys_reset_s)begin
      if(sys_reset_s == 1'b1)begin
	 start_daq_r <= 1'b0;
	 stop_daq_r <= 1'b0;
	 pause_daq_r <= 1'b0;
	 unconfigure_r <= 1'b0;
      end else begin
	 if(main_active_r == 1'b0)begin
	    start_daq_r <= 1'b0;
	    stop_daq_r <= 1'b0;
	    pause_daq_r <= 1'b0;
	    unconfigure_r <= 1'b0;
	 end else begin
	    
	    if(command_valid_s == 1'b1
	       && SIO_ADDR[11:0] == COMMAND)begin
	       if(SIO_WD[7:0] == COM_START)begin
		  start_daq_r <= 1'b1;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_STOP)begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b1;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_PAUSE)begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b1;
		  unconfigure_r <= 1'b0;
	       end else if(SIO_WD[7:0] == COM_UNCONFIGURE)begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b1;
	       end else begin
		  start_daq_r <= 1'b0;
		  stop_daq_r <= 1'b0;
		  pause_daq_r <= 1'b0;
		  unconfigure_r <= 1'b0;
	       end
	    end
	    
	 end // else: !if(main_active_r == 1'b0)
      end // else: !if(sys_reset_s == 1'b1)
   end // always @ (posedge REG_CLK or posedge sys_reset_s)

   
   //---------------------------------
   // Prevent to timing fluctuation
   //---------------------------------
   wire [3:0] or_beclk_s;

   assign beclk1_s = (clk_sel_r == 1'b1) ? 1'b1 : ~beclk_r;
   assign beclk2_s = (clk_sel_r == 1'b1) ? 1'b0 : ~beclk_r;
   
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk1(.Q(or_beclk_s[0]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk2(.Q(or_beclk_s[1]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk3(.Q(or_beclk_s[2]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );
   ODDR#(.DDR_CLK_EDGE("OPPOSITE_EDGE"), .INIT(1'b0), .SRTYPE("SYNC"))
   ODDR_beclk4(.Q(or_beclk_s[3]), .C(clk_40M_90_s), .CE(1'b1),
	      .D1(beclk1_s), .D2(beclk2_s), .R(1'b0), .S(1'b0) );

   (* IOB = "FORCE" *) reg [3:0] or_feclk_r;
   (* IOB = "FORCE" *) reg [3:0] or_prin_r;
   always @(posedge clk_40M_s)begin
      or_feclk_r[3:0] <= feclk2_r[3:0];
      or_prin_r[3:0] <= prin2_r[3:0];
   end
   
   (* IOB = "FORCE" *) reg [31:0] or_bus_r;
   (* IOB = "FORCE" *) reg [3:0]  or_chmode_r;
   (* IOB = "FORCE" *) reg [3:0]  or_bemode_r;
   (* IOB = "FORCE" *) reg [3:0]  or_femode_r;
   always @(posedge clk_40M_s)begin
      or_bus_r[31:0] <= bus_s[31:0];
      or_chmode_r[3:0] <= { 4{chmode_out_s} };
      or_bemode_r[3:0] <= { 4{bemode_out_s} };
      or_femode_r[3:0] <= { 4{femode_out_s} };
   end

   // CLK Delay
   wire [3:0] id_beclk_s;
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK0(.DATAOUT(id_beclk_s[0]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_beclk_s[0]), .RST(1'b0), .T(1'b0));
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK1(.DATAOUT(id_beclk_s[1]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_beclk_s[1]), .RST(1'b0), .T(1'b0));
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK2(.DATAOUT(id_beclk_s[2]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_beclk_s[2]), .RST(1'b0), .T(1'b0));
   IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   IODELAY_BCLK3(.DATAOUT(id_beclk_s[3]), .C(1'b0), .CE(1'b0),
   		 .DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_beclk_s[3]), .RST(1'b0), .T(1'b0));
   
   // BUS line IODELAY
   // wire [31:0] id_bus_s;
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[0]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[0]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[1]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[1]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[2]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[2]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[3]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[3]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[4]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[4]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[5]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[5]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[6]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[6]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[7]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[7]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[8]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[8]), .RST(sys_reset_s), .T(1'b0));
   // IODELAY #(.DELAY_SRC("O"), .HIGH_PERFORMANCE_MODE("TRUE"), .IDELAY_TYPE("FIXED"), .IDELAY_VALUE(0),
   // 	     .ODELAY_VALUE(0), .REFCLK_FREQUENCY(200.00), .SIGNAL_PATTERN("DATA"))
   // IODELAY_BUS0(.DATAOUT(id_bus_s[9]), .C(ODELAY_CLK), .CE(1'b0),
   // 		.DATAIN(1'b0), .IDATAIN(1'b0), .INC(1'b0), .ODATAIN(or_bus_r[9]), .RST(sys_reset_s), .T(1'b0));
   
   //telescope1
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS0_MAP(.O(bus_data_s[0]), .IO(BUS[0]), .I(or_bus_r[0]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS1_MAP(.O(bus_data_s[1]), .IO(BUS[1]), .I(or_bus_r[1]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS2_MAP(.O(bus_data_s[2]), .IO(BUS[2]), .I(or_bus_r[2]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS3_MAP(.O(bus_data_s[3]), .IO(BUS[3]), .I(or_bus_r[3]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS4_MAP(.O(bus_data_s[4]), .IO(BUS[4]), .I(or_bus_r[4]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS5_MAP(.O(bus_data_s[5]), .IO(BUS[5]), .I(or_bus_r[5]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS6_MAP(.O(bus_data_s[6]), .IO(BUS[6]), .I(or_bus_r[6]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS7_MAP(.O(bus_data_s[7]), .IO(BUS[7]), .I(or_bus_r[7]), .T(io_state_sel_s));
   
   //telescope2
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS8_MAP(.O(bus_data_s[8]), .IO(BUS[8]), .I(or_bus_r[8]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS9_MAP(.O(bus_data_s[9]), .IO(BUS[9]), .I(or_bus_r[9]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS10_MAP(.O(bus_data_s[10]), .IO(BUS[10]), .I(or_bus_r[10]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS11_MAP(.O(bus_data_s[11]), .IO(BUS[11]), .I(or_bus_r[11]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS12_MAP(.O(bus_data_s[12]), .IO(BUS[12]), .I(or_bus_r[12]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS13_MAP(.O(bus_data_s[13]), .IO(BUS[13]), .I(or_bus_r[13]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS14_MAP(.O(bus_data_s[14]), .IO(BUS[14]), .I(or_bus_r[14]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS15_MAP(.O(bus_data_s[15]), .IO(BUS[15]), .I(or_bus_r[15]), .T(io_state_sel_s));
   
   //telescope3
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS16_MAP(.O(bus_data_s[16]), .IO(BUS[16]), .I(or_bus_r[16]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS17_MAP(.O(bus_data_s[17]), .IO(BUS[17]), .I(or_bus_r[17]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS18_MAP(.O(bus_data_s[18]), .IO(BUS[18]), .I(or_bus_r[18]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS19_MAP(.O(bus_data_s[19]), .IO(BUS[19]), .I(or_bus_r[19]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS20_MAP(.O(bus_data_s[20]), .IO(BUS[20]), .I(or_bus_r[20]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS21_MAP(.O(bus_data_s[21]), .IO(BUS[21]), .I(or_bus_r[21]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS22_MAP(.O(bus_data_s[22]), .IO(BUS[22]), .I(or_bus_r[22]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS23_MAP(.O(bus_data_s[23]), .IO(BUS[23]), .I(or_bus_r[23]), .T(io_state_sel_s));
   
   //telescope4
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS24_MAP(.O(bus_data_s[24]), .IO(BUS[24]), .I(or_bus_r[24]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS25_MAP(.O(bus_data_s[25]), .IO(BUS[25]), .I(or_bus_r[25]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS26_MAP(.O(bus_data_s[26]), .IO(BUS[26]), .I(or_bus_r[26]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS27_MAP(.O(bus_data_s[27]), .IO(BUS[27]), .I(or_bus_r[27]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS28_MAP(.O(bus_data_s[28]), .IO(BUS[28]), .I(or_bus_r[28]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS29_MAP(.O(bus_data_s[29]), .IO(BUS[29]), .I(or_bus_r[29]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS30_MAP(.O(bus_data_s[30]), .IO(BUS[30]), .I(or_bus_r[30]), .T(io_state_sel_s));
   IOBUF #(.IOSTANDARD("LVCMOS33"))
   BUS31_MAP(.O(bus_data_s[31]), .IO(BUS[31]), .I(or_bus_r[31]), .T(io_state_sel_s));

   //---------------------------------------------------------//
   //telescope1
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE1_MAP(.O(CHMODE[0]), .I(or_chmode_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE1_MAP(.O(BEMODE[0]), .I(or_bemode_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE1_MAP(.O(FEMODE[0]), .I(or_femode_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK1_MAP(.O(FECLK[0]), .I(or_feclk_r[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK1_MAP(.O(BECLK[0]), .I(id_beclk_s[0]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin1_MAP(.O(PRin[0]), .I(or_prin_r[0]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV1_MAP(.O(obdv_s[0]), .I(OBDV[0]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout1_MAP(.O(prout_s[0]), .I(PRout[0]));
   
   //telescope2
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE2_MAP(.O(CHMODE[1]), .I(or_chmode_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE2_MAP(.O(BEMODE[1]), .I(or_bemode_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE2_MAP(.O(FEMODE[1]), .I(or_femode_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK2_MAP(.O(FECLK[1]), .I(or_feclk_r[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK2_MAP(.O(BECLK[1]), .I(id_beclk_s[1]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin2_MAP(.O(PRin[1]), .I(or_prin_r[1]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV2_MAP(.O(obdv_s[1]), .I(OBDV[1]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout2_MAP(.O(prout_s[1]), .I(PRout[1]));
   
   //telescope3
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE3_MAP(.O(CHMODE[2]), .I(or_chmode_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE3_MAP(.O(BEMODE[2]), .I(or_bemode_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE3_MAP(.O(FEMODE[2]), .I(or_femode_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK3_MAP(.O(FECLK[2]), .I(or_feclk_r[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK3_MAP(.O(BECLK[2]), .I(id_beclk_s[2]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin3_MAP(.O(PRin[2]), .I(or_prin_r[2]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV3_MAP(.O(obdv_s[2]), .I(OBDV[2]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout3_MAP(.O(prout_s[2]), .I(PRout[2]));
   
   //telescope4
   OBUF #(.IOSTANDARD("LVCMOS33"))
   CHMODE4_MAP(.O(CHMODE[3]), .I(or_chmode_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BEMODE4_MAP(.O(BEMODE[3]), .I(or_bemode_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FEMODE4_MAP(.O(FEMODE[3]), .I(or_femode_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   FECLK4_MAP(.O(FECLK[3]), .I(or_feclk_r[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   BECLK4_MAP(.O(BECLK[3]), .I(id_beclk_s[3]));
   OBUF #(.IOSTANDARD("LVCMOS33"))
   PRin4_MAP(.O(PRin[3]), .I(or_prin_r[3]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   OBDV4_MAP(.O(obdv_s[3]), .I(OBDV[3]));
   IBUF #(.IOSTANDARD("LVCMOS33"))
   PRout4_MAP(.O(prout_s[3]), .I(PRout[3]));
   
endmodule
